/**
 * @file ViewController.m
 * @brief TEPRA-Print SDK ViewController Class
 * @par Copyright:
 * (C) 2013-2019 KING JIM CO.,LTD.<BR>
 */

#import "ViewController.h"
#import "AppDelegate.h"
#import "TepraPrint.h"
#import "TepraPrintDataProvider.h"

#import "SampleDataProvider.h"
#import <UserNotifications/UserNotifications.h>

#define kFlatUIButtonColor  [UIColor colorWithRed:(0x00 / 255.0) green:(0x7A / 255.0) blue:(0xFF / 255.0) alpha:1.0]

static TepraPrint *_tepraPrint = nil;

@interface ViewController ()
{
    __strong NSDictionary *_printerInfo;
    __strong NSDictionary *_printSettngs;
    __strong NSArray *_formNames;
    __strong NSDictionary *_lwStatus;
    __strong NSTimer *_timer;
    NSInteger _jobNumber;
}

@end

@implementation ViewController

- (void)viewDidLoad
{
    [super viewDidLoad];
    _printerInfo = nil;
    _formNames = [NSArray arrayWithObject:@"Simple"];
    
    _printSettngs = [NSDictionary dictionaryWithObjectsAndKeys:
                     [NSNumber numberWithInteger:1],                @"Copies",
                     [NSNumber numberWithInteger:TapeCutEachLabel], @"Tape Cut",
                     [NSNumber numberWithBool:NO],                  @"Half Cut",
                     [NSNumber numberWithInteger:PrintSpeedHigh],   @"Print Speed",
                     [NSNumber numberWithInteger:0],                @"Density",
                     [NSNumber numberWithBool:NO],                  @"Priority Print",
                     [NSNumber numberWithBool:NO],                  @"Half Cut Continuous",
                     nil];
    
    _progressView.progress = 0.0;
    _processing = NO;
    
    [self setupViews];
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
}

- (void)viewWillAppear:(BOOL)animated
{
    if([_printerInfo objectForKey:@"BonjourName"] != nil) {
        [self.discoverButton setTitle:[NSString stringWithFormat:@"Select:%@", [_printerInfo objectForKey:@"BonjourName"]] forState:UIControlStateNormal];
    } else {
        [self.discoverButton setTitle:[NSString stringWithFormat:@"Select: Device is not selected"] forState:UIControlStateNormal];
    }
    
#ifdef __IPHONE_7_0
    double systemVersion = [[[UIDevice currentDevice] systemVersion] doubleValue];
	if ( systemVersion >= 7.0 )
    {
        [self updateViewSizeToFit:self.discoverButton];
    }
#endif
    
    _jobNumber = 0;
    [self updateFormButton];
    [super viewWillAppear:animated];
}

-(void)prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender
{
    UINavigationController *navi = [segue destinationViewController];
    if([[segue identifier] isEqualToString:@"DiscoverPrinter"]) {
        DiscoverTableViewController *view = (DiscoverTableViewController *)navi.topViewController;
        view.delegate = self;
    } else if([[segue identifier] isEqualToString:@"FormData"]){
        FormSelectTableViewController *view = (FormSelectTableViewController *)navi.topViewController;
        view.delegate = self;
        view.selectForms = _formNames;
        NSString* formListPath = [[NSBundle mainBundle] pathForResource:@"FormDataList" ofType:@"plist"];
        view.formDataList = [[NSArray alloc] initWithContentsOfFile:formListPath];
    } else if([[segue identifier] isEqualToString:@"PrintSettings"]){
        PrintSettingsViewController *view = (PrintSettingsViewController *)navi.topViewController;
        view.delegate = self;
        view.settings = _printSettngs;
        view.printerInfo = _printerInfo;
    }
}

- (IBAction)doPrint:(id)sender
{
    _processing = YES;
    _jobNumber = 0;
    [self performPrint];
}


- (IBAction)fetchStatus:(id)sender
{
    if(_printerInfo == nil) return;
    
    _processing = YES;
    _tepraStatusLabel.text = @"";
    _tapeLabel.text = @"";
    
    TepraPrint *tepraPrint = [ViewController sharedTepraPrint];
    
    [_statusIndicator startAnimating];
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT,0), ^{
        [tepraPrint setPrinterInformation:self->_printerInfo];
        self->_lwStatus = [tepraPrint fetchPrinterStatus];
        dispatch_async(dispatch_get_main_queue(), ^{
            [self->_statusIndicator stopAnimating];
            TepraPrintStatusError deviceError = [tepraPrint deviceErrorFromStatus:self->_lwStatus];
            self->_tepraStatusLabel.text = [NSString stringWithFormat:@"%0lX", (unsigned long)deviceError];
            TepraPrintTapeWidth tape = [tepraPrint tapeWidthFromStatus:self->_lwStatus];
            self->_tapeLabel.text = [self tapeWidthStringFromTapeWidhCode:tape];
            self->_processing = NO;
        });
    });

}

- (IBAction)tapeCut:(id)sender
{
    if(_printerInfo == nil) return;
    
    _processing = YES;
    _cancelButton.hidden = NO;
    [self adjustButtons:NO];
    [_feedIndicator startAnimating];

    TepraPrint *tepraPrint = [ViewController sharedTepraPrint];
    tepraPrint.delegate = self;
    [tepraPrint setPrinterInformation:_printerInfo];
    [tepraPrint doTapeFeed:TapeOperationFeedAndCut];
    
}
- (IBAction)tapeFeed:(id)sender
{
    if(_printerInfo == nil) return;
    
    _processing = YES;
    _cancelButton.hidden = NO;
    [self adjustButtons:NO];
    [_feedIndicator startAnimating];

    TepraPrint *tepraPrint = [ViewController sharedTepraPrint];
    tepraPrint.delegate = self;
    [tepraPrint setPrinterInformation:_printerInfo];
    [tepraPrint doTapeFeed:TapeOperationFeed];

}

- (IBAction)doCancel:(id)sender
{
    TepraPrint *tepraPrint = [ViewController sharedTepraPrint];
    tepraPrint.delegate = self;
    [tepraPrint cancelPrint];

}

#pragma mark - delegate
- (void)discoverView:(DiscoverTableViewController *)discoverView didSelectPrinter:(NSDictionary *)printerInfo
{
    _printerInfo = printerInfo;
}

- (void)formSelectView:(FormSelectTableViewController *)formSelectView didSelectForm:(NSArray *)formNames
{
    _formNames = formNames;
}

- (void)printSettingsView:(PrintSettingsViewController *)printSettingsView didEndSettings:(NSDictionary *)settings
{
    _printSettngs = settings;
}

//
- (void)tepraPrint:(TepraPrint *)tepraPrint didChangePrintOperationPhase:(TepraPrintPrintingPhase)jobPhase
{
    
    NSString *phase;
    switch (jobPhase) {
        case PrintingPhasePrepare:
            phase = @"PrintingPhasePrepare";
            break;
        case PrintingPhaseProcessing:
            phase = @"PrintingPhaseProcessing";
            break;
        case PrintingPhaseWaitingForPrint:
            phase = @"PrintingPhaseWaitingForPrint";
            [_feedIndicator startAnimating];
            break;
        case PrintingPhaseComplete:
            phase = @"PrintingPhaseComplete";
            [_feedIndicator stopAnimating];
            _progressView.progress = 0.0;
            [_timer invalidate];
            _timer = nil;
            _cancelButton.hidden = YES;
            [self adjustButtons:YES];
            
            _jobNumber++;
            [self performSelector:@selector(performPrint)];
            
            break;
            
        default:
            phase = @"";
            [_feedIndicator stopAnimating];
            _cancelButton.hidden = YES;
            [self adjustButtons:YES];
            
            break;
    }
    _printPhase.text = phase;
}

- (void)tepraPrint:(TepraPrint *)tepraPrint didChangeTapeFeedOperationPhase:(TepraPrintPrintingPhase)jobPhase
{
    
    NSString *phase;
    switch (jobPhase) {
        case PrintingPhasePrepare:
            phase = @"PrintingPhasePrepare";
            break;
        case PrintingPhaseProcessing:
            phase = @"PrintingPhaseProcessing";
            _cancelButton.hidden = YES;
            break;
        case PrintingPhaseWaitingForPrint:
            phase = @"PrintingPhaseWaitingForPrint";
            break;
        case PrintingPhaseComplete:
            phase = @"PrintingPhaseComplete";
            [_feedIndicator stopAnimating];
            _cancelButton.hidden = YES;
            [self adjustButtons:YES];
            _processing = NO;
            break;
            
        default:
            phase = @"";
            [_feedIndicator stopAnimating];
            _cancelButton.hidden = YES;
            [self adjustButtons:YES];
            break;
    }
    _printPhase.text = phase;
}

- (void)tepraPrint:(TepraPrint *)tepraPrint didAbortPrintOperation:(TepraPrintConnectionStatus)errorStatus deviceStatus:(TepraPrintStatusError)deviceStatus
{
    [self printComplete:errorStatus deviceStatus:deviceStatus isSuspend:NO];

    _cancelButton.hidden = YES;
    [self adjustButtons:YES];
    
    [_timer invalidate];
    _timer = nil;
    _progressView.progress = 0.0;
    _processing = NO;
    
    NSString *message = [NSString stringWithFormat:@"Error Status : %ld\nDevice Status : %02lX", (long)errorStatus, (unsigned long)deviceStatus];
    UIAlertController *ac = [UIAlertController alertControllerWithTitle:@"Print Error!"
                                                                message:message
                                                         preferredStyle:UIAlertControllerStyleAlert];
    UIAlertAction *action = [UIAlertAction actionWithTitle:@"OK"
                                                     style:UIAlertActionStyleCancel
                                                   handler:^(UIAlertAction *act) {
                                                       [tepraPrint cancelPrint];
                                                   }];
    [ac addAction:action];
    
    [self presentViewController:ac animated:YES completion:nil];
}

- (void)tepraPrint:(TepraPrint *)tepraPrint didSuspendPrintOperation:(TepraPrintConnectionStatus)errorStatus deviceStatus:(TepraPrintStatusError)deviceStatus;
{
    [self printComplete:errorStatus deviceStatus:deviceStatus isSuspend:YES];
    NSString *message = [NSString stringWithFormat:@"Error Status : %ld\nDevice Status : %02lX", (long)errorStatus, (unsigned long)deviceStatus];
    UIAlertController *ac = [UIAlertController alertControllerWithTitle:@"Print Error! re-print ?"
                                                                message:message
                                                         preferredStyle:UIAlertControllerStyleAlert];
    UIAlertAction *okaction = [UIAlertAction actionWithTitle:@"OK"
                                                     style:UIAlertActionStyleDefault
                                                   handler:^(UIAlertAction *act) {
                                                       [tepraPrint resumeOfPrint];
                                                   }];
    UIAlertAction *cancelaction = [UIAlertAction actionWithTitle:@"Cancel"
                                                     style:UIAlertActionStyleCancel
                                                   handler:^(UIAlertAction *act) {
                                                       [tepraPrint cancelPrint];
                                                   }];
    [ac addAction:okaction];
    [ac addAction:cancelaction];
    
    [self presentViewController:ac animated:YES completion:nil];
}

- (void)tepraPrint:(TepraPrint *)tepraPrint didAbortTapeFeedOperation:(TepraPrintConnectionStatus)errorStatus deviceStatus:(TepraPrintStatusError)deviceStatus
{
    [_feedIndicator stopAnimating];
    _cancelButton.hidden = YES;
    [self adjustButtons:YES];
    [_timer invalidate];
    _timer = nil;
    _progressView.progress = 0.0;
    _processing = NO;

    NSString *message = [NSString stringWithFormat:@"Error Status : %ld\nDevice Status : %02lX", (long)errorStatus, (unsigned long)deviceStatus];
    UIAlertController *ac = [UIAlertController alertControllerWithTitle:@"Tape Feed Error!"
                                                                message:message
                                                         preferredStyle:UIAlertControllerStyleAlert];
    UIAlertAction *cancelaction = [UIAlertAction actionWithTitle:@"OK"
                                                           style:UIAlertActionStyleCancel
                                                         handler:^(UIAlertAction *act) {
                                                             [tepraPrint cancelPrint];
                                                         }];
    [ac addAction:cancelaction];
    
    [self presentViewController:ac animated:YES completion:nil];
}

- (void)saveCurrentData
{
}

#pragma mark - private
- (void)setupViews
{
    double systemVersion = [[[UIDevice currentDevice] systemVersion] doubleValue];
	if ( systemVersion >= 7.0 )
    {
#ifdef __IPHONE_7_0
        CGFloat statusBarHeight = CGRectGetHeight([UIApplication sharedApplication].statusBarFrame);
        for ( UIView *subview in self.view.subviews )
        {
            CGRect frame = CGRectOffset(subview.frame, 0.0, statusBarHeight);
            subview.frame = frame;
        }
        
        [self updateButtons:self.view];
#else
        _progressView.frame = CGRectMake(CGRectGetMinX(_progressView.frame),
                                         CGRectGetMinY(_progressView.frame),
                                         CGRectGetWidth(_progressView.frame),
                                         9.0);
#endif
    }
}

- (void)updateButtons:(UIView *)superview
{
    for ( UIView *subview in superview.subviews )
    {
        if ( [subview isKindOfClass:[UIButton class]] == YES )
        {
            UIButton *button = (UIButton *)subview;
            button.titleLabel.font = [UIFont systemFontOfSize:15.0];
            [button setTitleColor:kFlatUIButtonColor forState:UIControlStateNormal];
            [button setTitleShadowColor:nil forState:UIControlStateNormal];
            
            [self updateViewSizeToFit:button];
        }
        else
        {
            [self updateButtons:subview];
        }
    }
}

- (void)updateViewSizeToFit:(UIView *)view
{
    CGRect frame = view.frame;
    
    [view sizeToFit];
    
    CGSize areaSize = CGSizeMake(CGRectGetMinX(frame) * 2 + CGRectGetWidth(frame),
                                 CGRectGetMinY(frame) * 2 + CGRectGetHeight(frame));
    frame = CGRectMake((areaSize.width - CGRectGetWidth(view.frame)) / 2,
                       (areaSize.height - CGRectGetHeight(view.frame)) / 2,
                       CGRectGetWidth(view.frame),
                       CGRectGetHeight(view.frame));
    view.frame = frame;
}

- (void)performPrint
{
    if([_formNames count] <= _jobNumber) {
        [self printComplete:ConnectionStatusNoError deviceStatus:StatusErrorNoError isSuspend:NO];
        _processing = NO;
        return;
    }
    
    if(_printerInfo == nil) return;
    
    [self updateFormButton];
    
    _cancelButton.hidden = NO;
    [self adjustButtons:NO];
    
    TepraPrint *tepraPrint = [ViewController sharedTepraPrint];
    [_statusIndicator startAnimating];
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT,0), ^{
        [tepraPrint setPrinterInformation:self->_printerInfo];
        self->_lwStatus = [tepraPrint fetchPrinterStatus];
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [self->_statusIndicator stopAnimating];
            
            if ([self->_lwStatus.allKeys containsObject:@"ST"]) {
                
                TepraPrintStatusError deviceError = [tepraPrint deviceErrorFromStatus:self->_lwStatus];
                self->_tepraStatusLabel.text = [NSString stringWithFormat:@"%0lX", (unsigned long)deviceError];
                TepraPrintTapeWidth tapeWidth = [tepraPrint tapeWidthFromStatus:self->_lwStatus];
                self->_tapeLabel.text = [self tapeWidthStringFromTapeWidhCode:tapeWidth];
                SampleDataProvider *provider = [[SampleDataProvider alloc] initWithFormName:[self->_formNames objectAtIndex:self->_jobNumber]];
                tepraPrint.delegate = self;
                
                NSDictionary *printParameter = [NSDictionary dictionaryWithObjectsAndKeys:
                                                [self->_printSettngs objectForKey:@"Copies"],    TepraPrintParameterKeyCopies,
                                                [self->_printSettngs objectForKey:@"Tape Cut"],  TepraPrintParameterKeyTapeCut,
                                                [self->_printSettngs objectForKey:@"Half Cut"],  TepraPrintParameterKeyHalfCut,
                                                [self->_printSettngs objectForKey:@"Print Speed"], TepraPrintParameterKeyPrintSpeed,
                                                [self->_printSettngs objectForKey:@"Density"],   TepraPrintParameterKeyDensity,
                                                [NSNumber numberWithInteger:tapeWidth],    TepraPrintParameterKeyTapeWidth,
                                                [self->_printSettngs objectForKey:@"Priority Print"], TepraPrintParameterKeyPriorityPrintSetting,
                                                [self->_printSettngs objectForKey:@"Half Cut Continuous"], TepraPrintParameterKeyHalfCutContinuous,
                                                nil];
                
                // Set the objects included in the print data.
                [tepraPrint setObjectType:ObjectTypeNone];
                //[tepraPrint setObjectType:ObjectTypeBarcode1D | ObjectTypeBarcode2D];
                
                [tepraPrint doPrint:provider printParameter:printParameter];
                
                self->_timer = [NSTimer scheduledTimerWithTimeInterval:1.0 target:self selector:@selector(updateProgress) userInfo:nil repeats:YES];
            } else {
                // プリンターステータス取得エラー
                [self printComplete:ConnectionStatusConnectionFailed deviceStatus:StatusErrorConnectionFailed isSuspend:YES];
                self->_cancelButton.hidden = YES;
                [self adjustButtons:YES];
                
                [self->_timer invalidate];
                self->_timer = nil;
                self->_progressView.progress = 0.0;
                self->_processing = NO;
                
                NSString *message = [NSString stringWithFormat:@"Can't get printer status."];
                UIAlertController *ac = [UIAlertController alertControllerWithTitle:@"Error!"
                                                                            message:message
                                                                     preferredStyle:UIAlertControllerStyleAlert];
                UIAlertAction *action = [UIAlertAction actionWithTitle:@"OK"
                                                                 style:UIAlertActionStyleCancel
                                                               handler:^(UIAlertAction *act) {
                    [tepraPrint cancelPrint];
                }];
                [ac addAction:action];
                
                [self presentViewController:ac animated:YES completion:nil];
            }
        });
    });
    
}

- (void)updateProgress
{
    TepraPrint *tepraPrint = [ViewController sharedTepraPrint];
    _progressView.progress = tepraPrint.progressOfPrint;
    _printingPage.text = [[NSNumber numberWithInteger:tepraPrint.pageNumberOfPrinting] stringValue];
}

- (NSString *)tapeWidthStringFromTapeWidhCode:(TepraPrintTapeWidth)tapeWidth
{
    static NSDictionary *_tapeStrings = 0;
    static dispatch_once_t onceToken = 0;
    dispatch_once(&onceToken, ^{
        _tapeStrings = [NSDictionary dictionaryWithObjectsAndKeys:
                        @"None",    @"0",
                        @"4mm",     @"1",
                        @"6mm",     @"2",
                        @"9mm",     @"3",
                        @"12mm",    @"4",
                        @"18mm",    @"5",
                        @"24mm",    @"6",
                        @"36mm",    @"7",
                        @"50mm",    @"12",
                        nil];
    });

    return [_tapeStrings objectForKey:[[NSNumber numberWithInteger:tapeWidth] stringValue]];
}

+ (TepraPrint *)sharedTepraPrint
{
    static dispatch_once_t onceToken = 0;
    dispatch_once(&onceToken, ^{
        _tepraPrint = [[TepraPrint alloc] init];
    });
    return _tepraPrint;

}

- (void)updateFormButton
{
/*== CHANGE 2014.08.25 Support Xcode5, 64bit, iOS SDK 7.1 */
//    [self.formSelectButton setTitle:[NSString stringWithFormat:@"Select:[%@] %d / %d", [_formNames objectAtIndex:_jobNumber], _jobNumber + 1, [_formNames count]] forState:UIControlStateNormal];
    [self.formSelectButton setTitle:[NSString stringWithFormat:@"Select:[%@] %d / %d", [_formNames objectAtIndex:_jobNumber], (int)(_jobNumber + 1), (int)[_formNames count]] forState:UIControlStateNormal];
/*== CHANGE End */
    
#ifdef __IPHONE_7_0
    double systemVersion = [[[UIDevice currentDevice] systemVersion] doubleValue];
	if ( systemVersion >= 7.0 )
    {
        [self updateViewSizeToFit:self.formSelectButton];
    }
#endif
}

- (void)adjustButtons:(BOOL)enable
{
    _printButton.enabled = enable;
    _tapeCutButton.enabled = enable;
    _tapeFeedBUtton.enabled = enable;
    _discoverButton.enabled = enable;
    _formSelectButton.enabled = enable;
    _printSettingsButton.enabled = enable;
}

- (void)printComplete:(TepraPrintConnectionStatus)connectionStatus deviceStatus:(TepraPrintStatusError)status isSuspend:(BOOL)suspend
{
	UIApplication*	app = [UIApplication sharedApplication];
    AppDelegate *appDelegate = (AppDelegate *)app.delegate;
    if(appDelegate.bgTask == UIBackgroundTaskInvalid) {
        return;
    }
        
	UIDevice* device = [UIDevice currentDevice];
	BOOL backgroundSupported = NO;
	if ([device respondsToSelector:@selector(isMultitaskingSupported)])
		backgroundSupported = device.multitaskingSupported;
	if(!backgroundSupported) return;
		
	NSString *msg;
	if(connectionStatus == ConnectionStatusNoError && status == StatusErrorNoError) {
		msg = @"Print Complete.";
	} else {
        if(suspend) {
            msg = [NSString stringWithFormat:@"Print Error Re-Print [%02lx].", (unsigned long)status];
        } else {
            msg = [NSString stringWithFormat:@"Print Error [%02lx].", (unsigned long)status];
        }
    }
    
    UNMutableNotificationContent *alarm = [UNMutableNotificationContent new];
    alarm.body = msg;
    alarm.sound = [UNNotificationSound defaultSound];
    
    UNTimeIntervalNotificationTrigger *trigger = [UNTimeIntervalNotificationTrigger triggerWithTimeInterval:0.1 repeats:false];
    UNNotificationRequest *request = [UNNotificationRequest requestWithIdentifier:@"printComplete" content:alarm trigger:trigger];
    
    UNUserNotificationCenter *center = [UNUserNotificationCenter currentNotificationCenter];
    [center addNotificationRequest:request withCompletionHandler:nil];
}

@end
