/**
 * @file PrintSettingsViewController.m
 * @brief TEPRA-Print SDK PrintSettingsViewController Class
 * @par Copyright:
 * (C) 2013-2019 KING JIM CO.,LTD.<BR>
 */

#import "PrintSettingsViewController.h"
#import "TepraPrint.h"
#import "TepraPrintDiscoverPrinter.h"

typedef NS_ENUM(NSInteger,PrintSpeedSwitch) {
    SpeedSwitchHigh = 0,
    SpeedSwitchMiddle = 1,
    SpeedSwitchLow = 2
};

@interface PrintSettingsViewController ()
{
    NSInteger   _copies;
    NSInteger   _tapeCut;
    BOOL        _halfCut;
    PrintSpeedSwitch   _speedSwitch;
    TepraPrintPrintSpeed   _printSpeed;
    NSInteger   _density;
    BOOL        _priorityPrint;
    BOOL        _halfCutContinuous;
    
}
@end

@implementation PrintSettingsViewController

- (id)initWithStyle:(UITableViewStyle)style
{
    self = [super initWithStyle:style];
    if (self) {
        // Custom initialization
    }
    return self;
}

- (void)viewDidLoad
{
    [super viewDidLoad];
    _copies = [[_settings objectForKey:@"Copies"] integerValue];
    _tapeCut = [[_settings objectForKey:@"Tape Cut"] integerValue];
    _halfCut = [[_settings objectForKey:@"Half Cut"] boolValue];
    _printSpeed = [[_settings objectForKey:@"Print Speed"] integerValue];
    _density = [[_settings objectForKey:@"Density"] integerValue];
    _priorityPrint = [[_settings objectForKey:@"Priority Print"] boolValue];
    _halfCutContinuous = [[_settings objectForKey:@"Half Cut Continuous"] boolValue];
    
    switch (_printSpeed) {
        case PrintSpeedHigh:
            _speedSwitch = SpeedSwitchHigh;
            break;
        case PrintSpeedMiddle:
            _speedSwitch = SpeedSwitchMiddle;
            break;
        case PrintSpeedLow:
            _speedSwitch = SpeedSwitchLow;
            break;
        default:
            break;
    }
}

-(BOOL)isPriorityPrintCellEnable
{
    NSString *modelName = _printerInfo[TepraPrintPrinterInfoMDL];
    return  [modelName isEqualToString:@"SR-R680"]    ||
            [modelName isEqualToString:@"SR-R980"]    ||
            [modelName isEqualToString:@"TEPRA PRO SR-R7900P"]    ||
            [modelName isEqualToString:@"SR-MK1"]    ||
            [modelName isEqualToString:@"SR-R2500P"];
}

-(BOOL)isHalfCutContinuousCellEnable
{
    NSString *modelName = _printerInfo[TepraPrintPrinterInfoMDL];
    return  [modelName isEqualToString:@"TEPRA PRO SR-R7900P"]    ||
            [modelName isEqualToString:@"TEPRA PRO SR5900P"]      ||
            [modelName isEqualToString:@"TEPRA PRO SR5900C"]      ||
            [modelName isEqualToString:@"TEPRA PRO SR5900GS"]     ||
            [modelName isEqualToString:@"SR-R680"]                ||
            [modelName isEqualToString:@"SR-R980"];
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

#pragma mark - Table view data source

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return 1;
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return 8;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    UITableViewCell *cell;
    if(indexPath.row == 0) {
        static NSString *CellIdentifier1 = @"Copies cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier1];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier1];
        }
        UITextField *copiesField = (UITextField*)[cell viewWithTag:1];
        copiesField.text = [[NSNumber numberWithInteger:_copies] stringValue];
        copiesField.delegate = self;
        
    } else if(indexPath.row == 1) {
        static NSString *CellIdentifier2 = @"TapeCut cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier2];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier2];
        }
        UISegmentedControl *tapeCutsegment = (UISegmentedControl*)[cell viewWithTag:1];
        tapeCutsegment.selectedSegmentIndex = _tapeCut;
        [tapeCutsegment addTarget:self action:@selector(tapeCutSegmentAction:event:) forControlEvents:UIControlEventValueChanged];

    } else if(indexPath.row == 2) {
        static NSString *CellIdentifier3 = @"HalfCut cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier3];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier3];
        }
        UISwitch *halfCutSwitch = (UISwitch*)[cell viewWithTag:1];
        halfCutSwitch.on = _halfCut;
        [halfCutSwitch addTarget:self action:@selector(halfCutAction:event:) forControlEvents:UIControlEventTouchUpInside];

    } else if(indexPath.row == 3) {
        static NSString *CellIdentifier4 = @"PrintSpeed cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier4];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier4];
        }
        UISegmentedControl *printSpeedsegment = (UISegmentedControl*)[cell viewWithTag:1];
        printSpeedsegment.selectedSegmentIndex = _speedSwitch;
        [printSpeedsegment addTarget:self action:@selector(printSpeedSegmentAction:event:) forControlEvents:UIControlEventValueChanged];

    } else if(indexPath.row == 4) {
        static NSString *CellIdentifier5 = @"Density cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier5];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier5];
        }
        UITextField *densityField = (UITextField*)[cell viewWithTag:2];
        densityField.text = [[NSNumber numberWithInteger:_density] stringValue];
        densityField.delegate = self;

    } else if(indexPath.row == 5) {
        static NSString *CellIdentifier6 = @"PriorityPrint cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier6];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier6];
        }
        UISwitch *priorityPrintSwitch = (UISwitch*)[cell viewWithTag:1];
        priorityPrintSwitch.on = _priorityPrint;
        cell.userInteractionEnabled = [self isPriorityPrintCellEnable];
        [priorityPrintSwitch setEnabled:[self isPriorityPrintCellEnable]];
        [priorityPrintSwitch addTarget:self action:@selector(priorityPrintAction:event:) forControlEvents:UIControlEventTouchUpInside];
        
    } else if(indexPath.row == 6) {
        static NSString *CellIdentifier7 = @"HalfCutContinuous cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier7];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier7];
        }
        UISwitch *halfCutContinuousSwitch = (UISwitch*)[cell viewWithTag:1];
        halfCutContinuousSwitch.on = _halfCutContinuous;
        cell.userInteractionEnabled = [self isHalfCutContinuousCellEnable];
        [halfCutContinuousSwitch setEnabled:[self isHalfCutContinuousCellEnable]];
        [halfCutContinuousSwitch addTarget:self action:@selector(halfCutContinuousAction:event:) forControlEvents:UIControlEventTouchUpInside];
        
    } else if(indexPath.row == 7) {
        static NSString *CellIdentifier8 = @"Version cell";
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier8];
        if (cell == nil) {
            cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier8];
        }
        UILabel *versionLabel = (UILabel*)[cell viewWithTag:2];
        versionLabel.text = [TepraPrint version];
        
    }
    
    return cell;
}

#pragma mark - Table view delegate

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
	[self.tableView deselectRowAtIndexPath:indexPath animated:YES];
}

#pragma mark - done

- (IBAction)doneButton:(id)sender
{
    [self.tableView endEditing:YES];

    switch (_speedSwitch) {
        case SpeedSwitchHigh:
            _printSpeed = PrintSpeedHigh;
            break;
        case SpeedSwitchMiddle:
            _printSpeed = PrintSpeedMiddle;
            break;
        case SpeedSwitchLow:
            _printSpeed = PrintSpeedLow;
            break;
        default:
            break;
    }
    
    NSDictionary *dict = [NSDictionary dictionaryWithObjectsAndKeys:
                            [NSNumber numberWithInteger:_copies],   @"Copies",
                            [NSNumber numberWithInteger:_tapeCut],  @"Tape Cut",
                            [NSNumber numberWithBool:_halfCut],     @"Half Cut",
                            [NSNumber numberWithInteger:_printSpeed],    @"Print Speed",
                            [NSNumber numberWithInteger:_density],  @"Density",
                            [NSNumber numberWithBool:_priorityPrint],    @"Priority Print",
                            [NSNumber numberWithBool:_halfCutContinuous],    @"Half Cut Continuous",
                          nil];
    if ( [self.delegate respondsToSelector:@selector(printSettingsView:didEndSettings:)] ) {
        [self.delegate printSettingsView:self didEndSettings:dict];
    }

    [self dismissViewControllerAnimated:YES completion:nil];
}

#pragma mark - action
- (void)tapeCutSegmentAction:(id)sender event:(id)event
{
    UISegmentedControl *segment = sender;
    _tapeCut = segment.selectedSegmentIndex;
}

- (void)halfCutAction:(id)sender event:(id)event
{
    UISwitch *control = sender;
    _halfCut = control.on;
}

- (void)printSpeedSegmentAction:(id)sender event:(id)event
{
    UISegmentedControl *segment = sender;
    _speedSwitch = segment.selectedSegmentIndex;
}

- (void)priorityPrintAction:(id)sender event:(id)event
{
    UISwitch *control = sender;
    _priorityPrint = control.on;
}

- (void)halfCutContinuousAction:(id)sender event:(id)event
{
    UISwitch *control = sender;
    _halfCutContinuous = control.on;
}

#pragma mark - delegate
- (BOOL)textFieldShouldReturn:(UITextField *)textField
{
    [textField resignFirstResponder];
    return YES;
}

- (void)textFieldDidEndEditing:(UITextField *)textField
{
    if(textField.tag == 1) {    // copies
        NSInteger value = [textField.text integerValue];
        if(value >= 1 && value < 100) {
            _copies = value;
        } else {
            UIAlertController *ac = [UIAlertController alertControllerWithTitle:@"Error"
                                                                        message:@"The range of a copy is to 1 to 99."
                                                                 preferredStyle:UIAlertControllerStyleAlert];
            UIAlertAction *action = [UIAlertAction actionWithTitle:@"OK"
                                                             style:UIAlertActionStyleCancel
                                                           handler:^(UIAlertAction *act) {
                                                           }];
            [ac addAction:action];
            
            [self presentViewController:ac animated:YES completion:nil];
            textField.text = [[NSNumber numberWithInteger:_copies] stringValue];
        }
    } else {                    // density
        NSInteger value = [textField.text integerValue];
        if(value >= -5 && value <= 5) {
            _density = value;
        } else {
            UIAlertController *ac = [UIAlertController alertControllerWithTitle:@"Error"
                                                                        message:@"The range of a density is to -5 to 5."
                                                                 preferredStyle:UIAlertControllerStyleAlert];
            UIAlertAction *action = [UIAlertAction actionWithTitle:@"OK"
                                                             style:UIAlertActionStyleCancel
                                                           handler:^(UIAlertAction *act) {
                                                           }];
            [ac addAction:action];
            
            [self presentViewController:ac animated:YES completion:nil];
            textField.text = [[NSNumber numberWithInteger:_density] stringValue];

        }
    }
}

@end
