/**
* @file ViewController.swift
* @brief TepraPrintSampleSwift ViewController Class definition
* @par Copyright:
* (C) 2018-2019 KING JIM CO.,LTD.<BR>
*/

import UIKit

class ViewController: UIViewController, TepraPrintDelegate, DiscoverTableViewControllerDelegate, UITextFieldDelegate, UIPickerViewDelegate, UIPickerViewDataSource
{
    // MARK: IBOutlet
    @IBOutlet weak var discoverButton: UIButton!
    @IBOutlet weak var dataTextField: UITextField!
    @IBOutlet weak var selectDataText: UITextField!
    @IBOutlet weak var printButton: UIButton!
    @IBOutlet weak var previewButton: UIButton!
    @IBOutlet weak var selectMarginText: UITextField!
    
    // MARK: private member
    fileprivate let SHARED_TEPRAPRINT: TepraPrint = TepraPrint()
    fileprivate let DATA_PROVIDER: SampleDataProvider = SampleDataProvider()
    
    fileprivate var processing: Bool = false
    fileprivate var printerInfo: [String: AnyObject]?
    
    let dataList = ["Text", "QRCode", "Img1", "Img2", "Imgs"];
    let marginList = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18", "19", "20", "21"];
    var pickerViewText: UIPickerView = UIPickerView()
    var pickerViewMargin: UIPickerView = UIPickerView()
    var selectData:DataType = .text
    var selectMargin:UInt = 1
    
    enum ProcessType
    {
        case print
        case preview
    }
    
    enum DataType
    {
        case text
        case qrcode
        case img1
        case img2
        case imgs
    }
    
    // MARK: override method
    override func viewDidLoad()
    {
        super.viewDidLoad()
        
        dataTextField.text = sharedDataProvider().stringData
        
        let toolbar = UIToolbar(frame: CGRect(x:0,y:0,width:0,height:35))
        let doneItem = UIBarButtonItem(barButtonSystemItem: .done, target: self, action: #selector(ViewController.done))
        toolbar.setItems([doneItem], animated: true)
        
        pickerViewText.delegate = self
        pickerViewText.dataSource = self
        pickerViewText.showsSelectionIndicator = true
        pickerViewText.tag = 1
        selectDataText.inputView = pickerViewText
        selectDataText.inputAccessoryView = toolbar
        selectDataText.text = dataList[0]
        
        pickerViewMargin.delegate = self
        pickerViewMargin.dataSource = self
        pickerViewMargin.showsSelectionIndicator = true
        pickerViewMargin.tag = 2
        selectMarginText.inputView = pickerViewMargin
        selectMarginText.inputAccessoryView = toolbar
        selectMarginText.text = marginList[0]
        
        selectMarginText.isHidden = true
        
    }
    
    @objc func done()
    {
        selectDataText.endEditing(true)
        selectMarginText.endEditing(true)
    }
    
    // MARK: UIPickerView
    func numberOfComponents(in pickerView: UIPickerView) -> Int {
        return 1
    }
    
    // MARK: UIPickerView
    func pickerView(_ pickerView: UIPickerView, numberOfRowsInComponent component: Int) -> Int {
        if (pickerView.tag == 1){
            return dataList.count
        }else{
            return marginList.count
        }
    }
    
    // MARK: UIPickerView
    func pickerView(_ pickerView: UIPickerView, titleForRow row: Int, forComponent component: Int) -> String? {
        if(pickerView.tag == 1){
            return dataList[row]
        }else{
            return marginList[row]
        }
    }
    
    // MARK: UIPickerView
    func pickerView(_ pickerView: UIPickerView, didSelectRow row: Int, inComponent component: Int) {
        if(pickerView.tag == 1){
            selectData = intToDataType(num: row)
            selectDataText.text = dataList[row]
            switch selectData {
            case .text:
                dataTextField.text = sharedDataProvider().stringData
                selectMarginText.isHidden = true
            case .qrcode:
                dataTextField.text = sharedDataProvider().qrCodeData
                selectMarginText.isHidden = true
            default:
                dataTextField.text = ""
                selectMarginText.isHidden = false
            }
        }else{
            if(row == 0){
                selectMargin = 1
                selectMarginText.text = marginList[row]
            }else{
                selectMargin = UInt(marginList[row])!
                selectMarginText.text = marginList[row]
            }
        }
    }
    
    
    func intToDataType(num: Int) -> DataType
    {
        let ret: DataType
        switch num {
        case 1:
            ret = .qrcode
        case 2:
            ret = .img1
        case 3:
            ret = .img2
        case 4:
            ret = .imgs
        default:
            ret = .text
        }
        return ret
    }
    
    override func viewWillAppear(_ animated: Bool)
    {
        super.viewWillAppear(animated)
        
        discoverButton.setTitle("Select: Device is not selected", for: UIControl.State())
        if let _printerInfo = printerInfo
        {
            if let _bonjourName = _printerInfo["BonjourName"] as? String
            {
                discoverButton.setTitle(String(format: "Select:%@", _bonjourName), for: UIControl.State())
            }
        }
    }
    
    override func prepare(for segue: UIStoryboardSegue, sender: Any?)
    {
        dataTextField.resignFirstResponder()
        if (segue.identifier == "preview")
        {
            let previewView = segue.destination as! PreviewViewController
            previewView.image = sender as? UIImage
        }
        if segue.identifier == "DiscoverPrinter"
        {
            let discoverView = segue.destination as! DiscoverTableViewController
            discoverView.delegate = self
        }
    }
    
    override func didReceiveMemoryWarning()
    {
        super.didReceiveMemoryWarning()
    }
    
    // MARK: IBAction
    @IBAction func doPrint(_ sender: AnyObject)
    {
        dataTextField.resignFirstResponder()
        setProcessing(true)
        
        performPrint(process: .print)
    }
    
    @IBAction func setCustomData(_ sender: AnyObject)
    {
        if selectData == .qrcode
        {
            sharedDataProvider().qrCodeData = dataTextField.text
        }
        else if selectData == .text
        {
            sharedDataProvider().stringData = dataTextField.text
        }
    }
    
    // MARK: IBAction
    @IBAction func doPreview(_ sender: AnyObject)
    {
        dataTextField.resignFirstResponder()
        setProcessing(true)
        
        performPrint(process: .preview)
    }
    
    // MARK: DiscoverTableViewControllerDelegate
    func discoverView(_ discoverView: DiscoverTableViewController, didSelectPrinter printerInfo: [String: AnyObject])
    {
        self.printerInfo = printerInfo
    }
    
    // MARK: TepraPrintDelegate
    func tepraPrint(_ tepraPrint: TepraPrint!, didChangePrintOperationPhase jobPhase: TepraPrintPrintingPhase)
    {
        // Report the change of a printing phase
        var phase = ""
        
        switch jobPhase
        {
        case TepraPrintPrintingPhase.PrintingPhasePrepare:
            phase = "PrintingPhasePrepare"
        case TepraPrintPrintingPhase.PrintingPhaseProcessing:
            phase = "PrintingPhaseProcessing"
        case TepraPrintPrintingPhase.PrintingPhaseWaitingForPrint:
            phase = "PrintingPhaseWaitingForPrint"
        case TepraPrintPrintingPhase.PrintingPhaseComplete:
            phase = "PrintingPhaseComplete"
            self.printComplete(TepraPrintConnectionStatus.ConnectionStatusNoError, deviceStatus: TepraPrintStatusError.StatusErrorNoError, isSuspend: false)
            self.setProcessing(false)
        @unknown default:
            break
        }
        
        print("<tepraPrint(_:didChangePrintOperationPhase:)>phase=\(phase)")
    }
    
    func tepraPrint(_ tepraPrint: TepraPrint!, didAbortPrintOperation errorStatus: TepraPrintConnectionStatus, deviceStatus: TepraPrintStatusError)
    {
        // It is called when undergoing a transition to the printing cancel operation due to a printing error
        self.printComplete(errorStatus, deviceStatus: deviceStatus, isSuspend: false)
        
        self.setProcessing(false)
        
        let message = String(format: "Error Status : %d\nDevice Status : %02X", errorStatus.rawValue, deviceStatus.rawValue)
        let alert: UIAlertController = UIAlertController(title: "Print Error!", message: message, preferredStyle: UIAlertController.Style.alert)
        let cancelAction : UIAlertAction = UIAlertAction(title: "OK", style: UIAlertAction.Style.default, handler:{(action: UIAlertAction!) -> Void in
            tepraPrint.cancel()
        })
        alert.addAction(cancelAction)
        
        // 既にアラートが表示済みの場合は閉じてから表示する
        if let presented = self.presentedViewController as? UIAlertController {
            presented.dismiss(animated: true, completion: {
                self.present(alert, animated: true, completion: nil)
            })
        } else {
            self.present(alert, animated: true, completion: nil)
        }
    }
    
    func tepraPrint(_ tepraPrint: TepraPrint!, didSuspendPrintOperation errorStatus: TepraPrintConnectionStatus, deviceStatus: TepraPrintStatusError)
    {
        // It is called when undergoing a transition to the printing restart operation due to a printing error
        self.printComplete(errorStatus, deviceStatus: deviceStatus, isSuspend: true)
        
        let message = String(format: "Error Status : %d\nDevice Status : %02X", errorStatus.rawValue, deviceStatus.rawValue)
        let alert: UIAlertController = UIAlertController(title: "Error!", message: message, preferredStyle: UIAlertController.Style.alert)
        let cancelAction : UIAlertAction = UIAlertAction(title: "Cancel", style: UIAlertAction.Style.default, handler:{(action: UIAlertAction!) -> Void in
            tepraPrint.cancel()
        })
        let defaultAction : UIAlertAction = UIAlertAction(title: "OK", style: UIAlertAction.Style.default, handler:{(action: UIAlertAction!) -> Void in
            tepraPrint.resumeOfPrint()
        })
        alert.addAction(cancelAction)
        alert.addAction(defaultAction)
        
        // 既にアラートが表示済みの場合は閉じてから表示する
        if let presented = self.presentedViewController as? UIAlertController {
            presented.dismiss(animated: true, completion: {
                self.present(alert, animated: true, completion: nil)
            })
        } else {
            self.present(alert, animated: true, completion: nil)
        }
    }
    
    // MARK: UITextFieldDelegate
    func textFieldDidEndEditing(_ textField: UITextField)
    {
        if selectData == .text
        {
            sharedDataProvider().stringData = dataTextField.text
        }
        else if selectData == .qrcode
        {
            sharedDataProvider().qrCodeData = dataTextField.text
        }
    }
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool
    {
        textField.resignFirstResponder()
        return true
    }
    
    // MARK: public method
    func getProcessing() -> Bool
    {
        return processing
    }
    
    // MARK: private method
    fileprivate func performPrint(process :ProcessType)
    {
        guard let printerInfo = self.printerInfo else
        {
            let message = String(format: "Device is not selected.")
            let alert: UIAlertController = UIAlertController(title: "Error!", message: message, preferredStyle: UIAlertController.Style.alert)
            let cancelAction : UIAlertAction = UIAlertAction(title: "OK", style: UIAlertAction.Style.default, handler:{(action: UIAlertAction!) -> Void in
                let tepraPrint = self.sharedTepraPrint()
                tepraPrint.cancel()
            })
            alert.addAction(cancelAction)
            
            present(alert, animated: true, completion: nil)
            
            setProcessing(false)
            return
        }
        
        DispatchQueue.global(qos: .default).async { () -> Void in
            
            let tepraPrint = self.sharedTepraPrint()
            var image: UIImage? = nil
            var images: [UIImage]? = nil
            
            // Set printing information
            tepraPrint.setPrinterInformation(printerInfo)
            
            // Obtain printing status
            let lwStatus = tepraPrint.fetchPrinterStatus()
            if lwStatus == nil || !(lwStatus?.keys.contains("ST") ?? false) {
                DispatchQueue.main.async(execute: { () -> Void in
                    // プリンターステータス取得エラー
                    let message = String(format: "Can't get printer status.")
                    let alert: UIAlertController = UIAlertController(title: "Error!", message: message, preferredStyle: UIAlertController.Style.alert)
                    let cancelAction : UIAlertAction = UIAlertAction(title: "OK", style: UIAlertAction.Style.default, handler:{(action: UIAlertAction!) -> Void in
                        tepraPrint.cancel()
                    })
                    alert.addAction(cancelAction)
                    
                    self.present(alert, animated: true, completion: nil)
                    
                    self.setProcessing(false)
                })
                return
            }
            
            DispatchQueue.main.async(execute: { () -> Void in
                
                let tapeWidth: TepraPrintTapeWidth = tepraPrint.tapeWidth(fromStatus: lwStatus)

                // テープ幅取得エラー
                if tapeWidth.rawValue < TepraPrintTapeWidth.TapeWidth4mm.rawValue || tapeWidth.rawValue > TepraPrintTapeWidth.TapeWidth50mmNew.rawValue {
                    let message = String(format: "Can't get tape width.")
                    let alert: UIAlertController = UIAlertController(title: "Error!", message: message, preferredStyle: UIAlertController.Style.alert)
                    let cancelAction: UIAlertAction = UIAlertAction(title: "OK", style: UIAlertAction.Style.default, handler:{(action: UIAlertAction!) -> Void in
                        tepraPrint.cancel()
                    })
                    
                    alert.addAction(cancelAction)
                    
                    self.present(alert, animated: true, completion: nil)
                    return
                }

                let dataProvider = self.sharedDataProvider()
                switch self.selectData
                {
                case .qrcode:
                    dataProvider.formType = .qrCode
                case .img1:
                    image = self.createImage(text: "Hello Tape:1", tapeWidth: tapeWidth)
                case .img2:
                    image = self.createImage(text: "Hello Tape:2", tapeWidth: tapeWidth)
                case .imgs:
                    if let img1 = self.createImage(text: "Hello Tape:1", tapeWidth: tapeWidth), let img2 = self.createImage(text: "Hello Tape:2", tapeWidth: tapeWidth) {
                        images = [img1, img2]
                    }
//                    images = []
//                    images! += [UIImage(named: "Img2_432x216.png")!,UIImage(named: "Img1_432x216.png")!]
                default:
                    dataProvider.formType = .string
                }
                
                // Make a print parameter
                let halfCut = tepraPrint.isSupportHalfCut()
                tepraPrint.delegate = self
                
                let printParameter: [String: AnyObject] = [
                    TepraPrintParameterKeyCopies: 1 as AnyObject,                                           // Number of copies(1 ... 99)
                    TepraPrintParameterKeyTapeCut: TepraPrintTapeCut.TapeCutEachLabel.rawValue as AnyObject,   // Tape cut method(TepraPrintTapeCut)
                    TepraPrintParameterKeyHalfCut: halfCut as AnyObject,                                    // Set half cut (isSupportHalfCut)
                    TepraPrintParameterKeyPrintSpeed: false as AnyObject,                                   // Low speed print setting (true:low speed print on)
                    TepraPrintParameterKeyDensity: 0 as AnyObject,                                          // Print density(-5 ... 5)
                    TepraPrintParameterKeyTapeWidth: tapeWidth.rawValue as AnyObject,                       // Tape width(TepraPrintTapeWidth)
                ]

                if (process == .print)
                {
                    // Carry out printing
                    if (image != nil) {
                        let data: Data! = image!.pngData()! as Data
                        tepraPrint.do(printParameter, print: data, margin:self.selectMargin)
                    } else if(images != nil){
                        var arr:[Data]! = []
                        for data in images! {
                            arr!.append(data.pngData()!)
                        }
                        tepraPrint.do(printParameter, printDataArray: arr, margin:self.selectMargin)
                    } else {
                        tepraPrint.do(dataProvider, printParameter: printParameter)
                    }
                } else {
                    // Carry out preview
                    if(images != nil){
                        image = images?[0]
                    }
                    if (image == nil) {
                        let images:NSArray = tepraPrint.labelImages(dataProvider, printParameter: printParameter)! as NSArray;
                        if (images.count > 0) {
                            image = images[0] as? UIImage;
                        }
                    }
                    self.performSegue(withIdentifier: "preview", sender: image)
                    self.setProcessing(false)
                }
            })
        }
    }
    
    fileprivate func sharedTepraPrint() -> TepraPrint
    {
        return SHARED_TEPRAPRINT
    }
    
    fileprivate func sharedDataProvider() -> SampleDataProvider
    {
        return DATA_PROVIDER
    }
    
    fileprivate func setProcessing(_ value: Bool)
    {
        processing = value
        
        if processing == true
        {
            discoverButton.isEnabled = false
            dataTextField.isEnabled = false
            printButton.isEnabled = false
            selectDataText.isEnabled = false
            previewButton.isEnabled = false
        }
        else
        {
            discoverButton.isEnabled = true
            dataTextField.isEnabled = true
            printButton.isEnabled = true
            selectDataText.isEnabled = true
            previewButton.isEnabled = true
        }
    }
    
    fileprivate func printComplete(_ connectionStatus: TepraPrintConnectionStatus, deviceStatus: TepraPrintStatusError, isSuspend: Bool)
    {
        let app = UIApplication.shared
        let appDelegate = app.delegate as! AppDelegate
        if appDelegate.bgTask == UIBackgroundTaskIdentifier.invalid
        {
            return
        }
        
        let device = UIDevice.current
        let backgroundSupported = device.isMultitaskingSupported
        if !backgroundSupported
        {
            return
        }

        var msg = ""
        if connectionStatus == TepraPrintConnectionStatus.ConnectionStatusNoError && deviceStatus == TepraPrintStatusError.StatusErrorNoError
        {
            msg = "Print Complete."
        }
        else
        {
            if isSuspend
            {
                msg = String(format: "Print Error Re-Print [%02x].", deviceStatus.rawValue)
            }
            else
            {
                msg = String(format: "Print Error [%02x].", deviceStatus.rawValue)
            }
        }
        
        let alarm = UNMutableNotificationContent()
        alarm.body = msg
        alarm.sound = .default
        
        let trigger = UNTimeIntervalNotificationTrigger(timeInterval: 0.1, repeats: false)
        let request = UNNotificationRequest(identifier: "printComplete", content: alarm, trigger: trigger)
         
        UNUserNotificationCenter.current().add(request)
    }
    
    private func createImage(text: String, tapeWidth: TepraPrintTapeWidth) -> UIImage?
    {
        
        let tepraPrint = self.sharedTepraPrint()
        let area = tepraPrint.printableSize(fromTape: tapeWidth)
        let height = CGFloat(area)
        let resolution = tepraPrint.resolution()
        let margin = CGFloat(self.selectMargin) / 25.4 * CGFloat(resolution)
        let ovalSize = height
        let attr = [.foregroundColor : UIColor.black,
                    .font : UIFont.systemFont(ofSize: height * 0.8)] as [NSAttributedString.Key : Any]
        let attrString = NSAttributedString(string: text, attributes: attr)
        let textSize = attrString.size()
        let width = margin + ovalSize + textSize.width + margin

        UIGraphicsBeginImageContext(CGSize(width: width, height: height))
        UIColor.white.setFill()
        UIBezierPath(rect: CGRect(x: 0, y: 0, width: width, height: height)).fill()
        UIColor.black.setStroke()
        UIBezierPath(ovalIn: CGRect(x: margin, y: 0, width: ovalSize, height: ovalSize)).stroke()
        attrString.draw(in: CGRect(x: margin + ovalSize, y: 0, width: textSize.width, height: height))
        
        return UIGraphicsGetImageFromCurrentImageContext()
    }

}

