/*
 * MainActivity.java
 *
 * Project: TEPRA-Print SDK
 *
 * Contains: MainActivity class
 *           SampleDataProvider class
 *           PrintCallback class
 *
 * (C) 2016-2019 KING JIM CO.,LTD.
 */
package jp.co.kingjim.tepraprint.sdk.demo;

import android.Manifest;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Notification;
import android.app.PendingIntent;
import android.app.ProgressDialog;
import android.bluetooth.BluetoothAdapter;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.TextView;
import android.widget.Toast;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import jp.co.kingjim.tepraprint.sdk.TepraPrint;
import jp.co.kingjim.tepraprint.sdk.TepraPrintCallback;
import jp.co.kingjim.tepraprint.sdk.TepraPrintConnectionStatus;
import jp.co.kingjim.tepraprint.sdk.TepraPrintDataProvider;
import jp.co.kingjim.tepraprint.sdk.TepraPrintDiscoverPrinter;
import jp.co.kingjim.tepraprint.sdk.TepraPrintParameterKey;
import jp.co.kingjim.tepraprint.sdk.TepraPrintPrintingPhase;
import jp.co.kingjim.tepraprint.sdk.TepraPrintStatusError;
import jp.co.kingjim.tepraprint.sdk.TepraPrintTapeOperation;
import jp.co.kingjim.tepraprint.sdk.TepraPrintTapeWidth;

public class MainActivity extends Activity implements OnClickListener {

	private enum ErrorMessage {
		CANT_GET_PRINTER_STATUS("Can't get printer status."),
		CANT_GET_TAPE_WIDTH("Can't get tape width.");

		private final String message;

		ErrorMessage(String message) {
			this.message = message;
		}

		@Override
		public String toString() {
			return message;
		}
	}

	private final String TAG = getClass().getSimpleName();

	private static final int NOTIFICATION_ID = 1;

	private static final int REQUEST_ENABLE_BT = 0;
	private static final int REQUEST_ACTIVITY_SEARCH = 1;
	private static final int REQUEST_ACTIVITY_SELECT_XML = 2;
	private static final int REQUEST_ACTIVITY_PRINT_SETTINGS = 3;

	private static final int REQUEST_PERMISSIONS = 1;

	private static final String KEY_FORMDATA = "formdata";
	private static final String SUFFIX = ".plist";

	TepraPrint tepraPrint;
	PrintCallback printListener;

	private Button buttonSearch;
	private Button buttonSelectXml;
	private ArrayList<String> _formNames = null;
	private Button buttonPrintSettings;
	private Button buttonPrinterStatus;
	private TextView textPrinterStatus;
	private TextView textTapeLabel;
	private Button buttonPrint;
	private Button buttonTapeCut;
	private Button buttonTapeFeed;
	private TextView textResult;
	private TextView textPrintingPage;

	private boolean _processing = false;
	private ProgressDialog progressDialog;
	private ProgressDialog waitDialog;
	private int _jobNumber = 0;
	Timer timer;

	Map<String, String> _printerInfo = null;
	Map<String, Object> _printSettings = null;
	Map<String, Integer> _tepraStatus = null;

	android.os.Handler handler = new android.os.Handler();

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);

		// Keep screen on
		getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);

		tepraPrint = new TepraPrint(this);
		tepraPrint.setCallback(printListener = new PrintCallback());

		buttonSearch = (Button) findViewById(R.id.search_button);
		buttonSearch.setOnClickListener(this);

		buttonSelectXml = (Button) findViewById(R.id.select_xml_button);
		buttonSelectXml.setOnClickListener(this);

		buttonPrintSettings = (Button) findViewById(R.id.print_settings_button);
		buttonPrintSettings.setOnClickListener(this);

		buttonPrinterStatus = (Button) findViewById(R.id.printer_status_button);
		buttonPrinterStatus.setOnClickListener(this);
		textPrinterStatus = (TextView) findViewById(R.id.printer_status_text_view);
		textTapeLabel = (TextView) findViewById(R.id.tape_label_text_view);

		buttonPrint = (Button) findViewById(R.id.print_button);
		buttonPrint.setOnClickListener(this);

		buttonTapeCut = (Button) findViewById(R.id.tape_cut_button);
		buttonTapeCut.setOnClickListener(this);

		buttonTapeFeed = (Button) findViewById(R.id.tape_feed_button);
		buttonTapeFeed.setOnClickListener(this);

		textResult = (TextView) findViewById(R.id.result_text_view);
		textPrintingPage = (TextView) findViewById(R.id.printingpage_text_view);

		// ProgressDialog for printing
		createProgressDialogForPrinting();

		// ProgressDialog
		waitDialog = new ProgressDialog(this);
		waitDialog.setMessage("Now processing...");
		waitDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
		waitDialog.setCancelable(false);

		// Print Settings
		setPrintSettingsValues();

		// Form Data
		if (TepraPrintSampleUtils.SAVE_VALUES_MODE) {
			ArrayList<String> savedFormData = TepraPrintSampleUtils.loadValues(
					KEY_FORMDATA, this);
			if (savedFormData.size() > 0) {
				_formNames = savedFormData;
				String s = getResources().getString(
						R.string.button_select_xml)
						+ "["
						+ savedFormData.get(_jobNumber)
						+ "] "
						+ (_jobNumber + 1) + " / " + savedFormData.size();
				buttonSelectXml.setText(s);
			}
		} else {
			buttonSelectXml.setText(getResources().getString(R.string.button_select_xml_no));
		}

		// Bluetooth
		try {
			BluetoothAdapter btAdapter = BluetoothAdapter.getDefaultAdapter();
			if (btAdapter == null) {
				Toast.makeText(getApplicationContext(),
						"Bluetooth is not available.", Toast.LENGTH_SHORT).show();
			} else {
				if (!btAdapter.isEnabled()) {
					Intent intent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
					if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
						if (checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT) == PackageManager.PERMISSION_GRANTED) {
							startActivityForResult(intent, REQUEST_ENABLE_BT);
						}
					} else {
						startActivityForResult(intent, REQUEST_ENABLE_BT);
					}
				}
			}
		} catch (Exception e) {
			Logger.w(TAG, "", e);
		}
	}

	@Override
	protected void onStart() {
		super.onStart();

		ArrayList<String> permissions = new ArrayList<>();

		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) { // API33 or higher
			// BLUETOOTH AUTH = BLUETOOTH_CONNECT
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_CONNECT);
			}
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_SCAN) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_SCAN);
			}
			// WIFI AUTH = NEARBY_WIFI_DEVICES
			if (checkSelfPermission(Manifest.permission.NEARBY_WIFI_DEVICES) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.NEARBY_WIFI_DEVICES);
			}
			// NOTIFICATION AUTH
			if (checkSelfPermission(Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.POST_NOTIFICATIONS);
			}
		} else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) { // API31,32
			// BLUETOOTH AUTH = BLUETOOTH_CONNECT
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_CONNECT);
			}
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_SCAN) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_SCAN);
			}
			// WIFI AUTH = LOCATION
			if (checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_FINE_LOCATION);
			}
			if (checkSelfPermission(Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_COARSE_LOCATION);
			}
		} else { // API30 or lower
			// BLUETOOTH AUTH & WIFI AUTH = LOCATION
			if (checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_FINE_LOCATION);
			}
			if (checkSelfPermission(Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_COARSE_LOCATION);
			}
		}

		if (!permissions.isEmpty()) {
			requestPermissions(permissions.toArray(new String[0]), REQUEST_PERMISSIONS);
		}
	}

	@Override
	public void onRequestPermissionsResult(int requestCode,
	                                       String[] permissions, int[] grantResults) {
		switch (requestCode) {
			case REQUEST_PERMISSIONS:
				boolean isGranted = true;
				for (int grantResult : grantResults) {
					if (grantResult != PackageManager.PERMISSION_GRANTED) {
						isGranted = false;
						break;
					}
				}
				if (!isGranted) {
					Toast.makeText(getApplicationContext(),
							"Since permission has not been granted, this app will not be able to discover the Bluetooth printer.", Toast.LENGTH_SHORT).show();
				}
				break;
		}
	}

	private void setPrintSettingsValues() {
		int copies = TepraPrintSampleUtils.DEFAULT_COPIES_SETTING;
		int tapeCut = TepraPrintSampleUtils.DEFAULT_TAPE_CUT_SETTING;
		boolean halfCut = tepraPrint.isSupportHalfCut();
		int printSpeed = TepraPrintSampleUtils.DEFAULT_PRINT_SPEED_SETTING;
		int density = TepraPrintSampleUtils.DEFAULT_DENSITY_SETTING;
		boolean printPriority = TepraPrintSampleUtils.DEFAULT_PRINT_PRIORITY_SETTING;
		boolean halfCutContinuous = TepraPrintSampleUtils.DEFAULT_HALF_CUT_COUNINUOUS_SETTING;

		_printSettings = new HashMap<String, Object>();
		_printSettings.put(TepraPrintParameterKey.Copies, copies);
		_printSettings.put(TepraPrintParameterKey.TapeCut, tapeCut);
		_printSettings.put(TepraPrintParameterKey.HalfCut, halfCut);
		_printSettings.put(TepraPrintParameterKey.PrintSpeed, printSpeed);
		_printSettings.put(TepraPrintParameterKey.Density, density);
		_printSettings.put(TepraPrintParameterKey.PriorityPrintSetting, printPriority);
		_printSettings.put(TepraPrintParameterKey.HalfCutContinuous, halfCutContinuous);

		SharedPreferences pref = getSharedPreferences(
				TepraPrintSampleUtils.PREFERENCE_FILE_NAME, MODE_PRIVATE);
		SharedPreferences.Editor editor = pref.edit();
		editor.putInt(TepraPrintParameterKey.Copies, copies);
		editor.putInt(TepraPrintParameterKey.TapeCut, tapeCut);
		editor.putBoolean(TepraPrintParameterKey.HalfCut, halfCut);
		editor.putInt(TepraPrintParameterKey.PrintSpeed, printSpeed);
		editor.putInt(TepraPrintParameterKey.Density, density);
		editor.putBoolean(TepraPrintParameterKey.PriorityPrintSetting, printPriority);
		editor.putBoolean(TepraPrintParameterKey.HalfCutContinuous, halfCutContinuous);
		editor.commit();
	}

	private void createProgressDialogForPrinting() {
		if (progressDialog == null) {
			progressDialog = new ProgressDialog(this);
			progressDialog.setMessage("Now printing...");
			progressDialog.setProgressStyle(ProgressDialog.STYLE_HORIZONTAL);
			progressDialog.setCancelable(false);
			progressDialog.setMax(100);
			progressDialog.incrementProgressBy(0);
			progressDialog.setButton(DialogInterface.BUTTON_NEGATIVE, "Cancel",
					new DialogInterface.OnClickListener() {
						public void onClick(DialogInterface dialog, int which) {
							progressDialog.setProgress(0);
							dialog.cancel();
						}
					});
			progressDialog
					.setOnCancelListener(new DialogInterface.OnCancelListener() {
						public void onCancel(DialogInterface dialog) {
							doCancel();
						}
					});
		}
	}

	@Override
	public void onClick(View v) {
		Intent intent = null;
		int viewId = v.getId();

		if (viewId == R.id.search_button) {
			intent = new Intent(this, SearchActivity.class);
			startActivityForResult(intent, REQUEST_ACTIVITY_SEARCH);
		} else if (viewId == R.id.select_xml_button) {
			intent = new Intent(this, SelectXmlActivity.class);
			if (_formNames != null) {
				intent.putStringArrayListExtra(KEY_FORMDATA, _formNames);
			}
			startActivityForResult(intent, REQUEST_ACTIVITY_SELECT_XML);
		} else if (viewId == R.id.print_settings_button) {
			intent = new Intent(this, PrintSettingsActivity.class);
			if (tepraPrint != null && _printerInfo != null) {
				tepraPrint.setPrinterInformation(_printerInfo);
				intent.putExtra("SupportHalfCut", tepraPrint.isSupportHalfCut());
				intent.putExtra("SupportPriorityPrint", isSupportPriorityPrint());
				intent.putExtra("SupportHalfCutContinuous", isSupportHalfCutContinuous());
			} else {
				intent.putExtra("SupportHalfCut", false);
				intent.putExtra("SupportPriorityPrint", false);
				intent.putExtra("SupportHalfCutContinuous", false);
			}
			startActivityForResult(intent, REQUEST_ACTIVITY_PRINT_SETTINGS);
		} else if (viewId == R.id.printer_status_button) {
			if (_printerInfo == null) {
				Toast.makeText(getApplicationContext(),
						"Device is not selected.", Toast.LENGTH_SHORT).show();
			} else {
				_processing = true;
				textPrinterStatus.setText("");
				textTapeLabel.setText("");

				waitDialog.show();

				ExecutorService executor = Executors.newSingleThreadExecutor();
				executor.execute(new Runnable() {
					@Override
					public void run() {
						tepraPrint.setPrinterInformation(_printerInfo);
						_tepraStatus = tepraPrint.fetchPrinterStatus();
						final PrinterStatus status = new PrinterStatus();
						status.setDeviceError(tepraPrint
								.getDeviceErrorFromStatus(_tepraStatus));
						status.setTapeWidth(tepraPrint
								.getTapeWidthFromStatus(_tepraStatus));


						new Handler(getMainLooper()).post(new Runnable() {
							@Override
							public void run() {
								notifyPrinterStatus(status.getDeviceError(),
										status.getTapeWidth());

								_processing = false;
								waitDialog.dismiss();
							}
						});

					}
				});
			}
		} else if (viewId == R.id.print_button) {
			if (_printerInfo == null) {
				Toast.makeText(getApplicationContext(),
						"Device is not selected.", Toast.LENGTH_SHORT).show();
			} else if (_formNames == null) {
				Toast.makeText(getApplicationContext(),
						"Form data is not selected.", Toast.LENGTH_SHORT).show();
			} else {
				_processing = true;
				_jobNumber = 0;
				performPrint();
			}
		} else if (viewId == R.id.tape_cut_button) {
			if (_printerInfo == null) {
				Toast.makeText(getApplicationContext(),
						"Device is not selected.", Toast.LENGTH_SHORT).show();
			} else {
				_processing = true;
				textResult.setText("");
				adjustButtons(false);
				waitDialog.show();

				tepraPrint.setPrinterInformation(_printerInfo);
				tepraPrint.doTapeFeed(TepraPrintTapeOperation.FeedAndCut);
			}
		} else if (viewId == R.id.tape_feed_button) {
			if (_printerInfo == null) {
				Toast.makeText(getApplicationContext(),
						"Device is not selected.", Toast.LENGTH_SHORT).show();
			} else {
				_processing = true;
				textResult.setText("");
				adjustButtons(false);
				waitDialog.show();

				tepraPrint.setPrinterInformation(_printerInfo);
				tepraPrint.doTapeFeed(TepraPrintTapeOperation.Feed);
			}
		}
	}

	private void notifyPrinterStatus(final int deviceError, final int tape) {
		handler.postDelayed(new Runnable() {
			public void run() {
				textPrinterStatus.setText(String.valueOf(deviceError));
				textTapeLabel.setText(getTapeWidthStringFromTapeWidhCode(tape));
			}
		}, 1);
	}

	private String getTapeWidthStringFromTapeWidhCode(int tapeWidth) {
		Map<Integer, String> tapeStrings = new HashMap<Integer, String>();
		tapeStrings.put(TepraPrintTapeWidth.None, "None");
		tapeStrings.put(TepraPrintTapeWidth.Normal_4mm, "4mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_6mm, "6mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_9mm, "9mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_12mm, "12mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_18mm, "18mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_24mm, "24mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_36mm, "36mm");
		tapeStrings.put(TepraPrintTapeWidth.Normal_New50mm, "50mm");
		tapeStrings.put(TepraPrintTapeWidth.Unknown, "Unknown");

		return tapeStrings.get(tapeWidth);
	}

	private void doCancel() {
		tepraPrint.cancelPrint();
	}

	private void performPrint() {
		if (_formNames == null) {
			return;
		}
		if (_formNames.size() <= _jobNumber) {
			_jobNumber = 0;
			printComplete(TepraPrintConnectionStatus.NoError, TepraPrintStatusError.NoError, false);
			_processing = false;
			return;
		}

		if (_printerInfo == null) {
			return;
		}

		updateFormButton();

		adjustButtons(false);

		Message msg = new Message();
		msg.obj = "";
		resultHandler.sendMessage(msg);

		handler.postDelayed(new Runnable() {
			public void run() {
				if (progressDialog != null) {
					progressDialog.show();
				}
			}
		}, 1);

		ExecutorService executor = Executors.newSingleThreadExecutor();
		executor.execute(new Runnable() {
			@Override
			public void run() {
				Boolean printResult;

				tepraPrint.setPrinterInformation(_printerInfo);
				_tepraStatus = tepraPrint.fetchPrinterStatus();

				int deviceError = tepraPrint.getDeviceErrorFromStatus(_tepraStatus);
				int tapeWidth = tepraPrint.getTapeWidthFromStatus(_tepraStatus);
				notifyPrinterStatus(deviceError, tapeWidth);

				if (_tepraStatus.isEmpty() || (deviceError == TepraPrintStatusError.ConnectionFailed)) {
					printResult = false;
				} else {
					if (tapeWidth < TepraPrintTapeWidth.Normal_4mm || tapeWidth > TepraPrintTapeWidth.Normal_New50mm) {
						// テープ幅取得エラー
						String message = ErrorMessage.CANT_GET_TAPE_WIDTH.message;
						alertAbortOperation("Error!", message);
						return;
					}

					SampleDataProvider provider = new SampleDataProvider(_formNames.get(_jobNumber) + SUFFIX);

					Map<String, Object> printParameter = new HashMap<String, Object>();
					printParameter.put(TepraPrintParameterKey.Copies,
							_printSettings.get(TepraPrintParameterKey.Copies));
					printParameter.put(TepraPrintParameterKey.TapeCut,
							_printSettings.get(TepraPrintParameterKey.TapeCut));
					printParameter.put(TepraPrintParameterKey.HalfCut,
							_printSettings.get(TepraPrintParameterKey.HalfCut));
					printParameter.put(TepraPrintParameterKey.PrintSpeed,
							_printSettings.get(TepraPrintParameterKey.PrintSpeed));
					printParameter.put(TepraPrintParameterKey.Density,
							_printSettings.get(TepraPrintParameterKey.Density));
					printParameter.put(TepraPrintParameterKey.TapeWidth, tapeWidth);
					printParameter.put(TepraPrintParameterKey.PriorityPrintSetting,
							_printSettings.get(TepraPrintParameterKey.PriorityPrintSetting));
					printParameter.put(TepraPrintParameterKey.HalfCutContinuous,
							_printSettings.get(TepraPrintParameterKey.HalfCutContinuous));
					tepraPrint.doPrint(provider, printParameter);

					printResult = true;
				}

				final Boolean result = printResult;
				if (!result) {
					new Handler(Looper.getMainLooper()).post(new Runnable() {
						@Override
						public void run() {
							waitDialog.dismiss();

							if (timer != null) {
								timer.cancel();
								timer = null;
							}
							if (progressDialog != null) {
								progressDialog.setProgress(0);
								progressDialog.dismiss();
								progressDialog = null;
							}
							runProgressDialogForPrinting();
							adjustButtons(true);
							_processing = false;

							String message = ErrorMessage.CANT_GET_PRINTER_STATUS.message;
							alertAbortOperation("Error", message);
						}
					});
				} else {
					if (timer != null) {
						timer.cancel();
						timer = null;
					}
					timer = new Timer();
					TimerTask task = (new TimerTask() {
						@Override
						public void run() {
							handler.post(new Runnable() {
								@Override
								public void run() {
									float progress = tepraPrint.getProgressOfPrint();
									if (progressDialog != null) {
										progressDialog.setProgress((int) (progress * 100));
									}
									int printingPage = tepraPrint.getPageNumberOfPrinting();
									textPrintingPage.setText(String.valueOf(printingPage));
								}
							});
						}
					});
					timer.schedule(task, 1000, 1000);
				}
			}
		});
	}

	@Override
	public void onDestroy() {
		if (timer != null) {
			timer.cancel();
			timer = null;
		}
		// Keep screen off
		getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
		super.onDestroy();
	}

	private void updateFormButton() {
		handler.postDelayed(new Runnable() {
			public void run() {
				if (_formNames != null) {
					String s = getResources().getString(
							R.string.button_select_xml)
							+ "["
							+ _formNames.get(_jobNumber)
							+ "] "
							+ (_jobNumber + 1) + " / " + _formNames.size();
					buttonSelectXml.setText(s);
				} else {
					String s = getResources().getString(
							R.string.button_select_xml)
							+ "[Nothing]";
					buttonSelectXml.setText(s);
				}
			}
		}, 1);
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);
		switch (requestCode) {
			case REQUEST_ENABLE_BT:
				if (resultCode != RESULT_OK) {
					Toast.makeText(getApplicationContext(),
							"Bluetooth is not enabled.", Toast.LENGTH_SHORT).show();
				}
				break;
			case REQUEST_ACTIVITY_SEARCH:
				if (resultCode == RESULT_OK) {
					Bundle extras = data.getExtras();
					if (extras != null) {
						if (_printerInfo != null) {
							_printerInfo.clear();
							_printerInfo = null;
						}
						_printerInfo = new HashMap<String, String>();
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_NAME,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_NAME));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_PRODUCT,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_PRODUCT));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_USBMDL,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_USBMDL));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_HOST,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_HOST));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_PORT,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_PORT));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_TYPE,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_TYPE));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_DOMAIN,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_DOMAIN));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_SERIAL_NUMBER,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_SERIAL_NUMBER));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_CLASS,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_CLASS));
						_printerInfo
								.put(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_STATUS,
										extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_STATUS));
						String s = getResources().getString(R.string.button_search) + extras.getString("name");
						buttonSearch.setText(s);
					}
				}
				break;
			case REQUEST_ACTIVITY_SELECT_XML:
				if (resultCode == RESULT_OK) {
					Bundle extras = data.getExtras();
					if (extras != null) {
						_formNames = extras.getStringArrayList(KEY_FORMDATA);
						String s = getResources().getString(
								R.string.button_select_xml)
								+ "["
								+ _formNames.get(_jobNumber)
								+ "] "
								+ (_jobNumber + 1) + " / " + _formNames.size();
						buttonSelectXml.setText(s);
					}
				}
				break;
			case REQUEST_ACTIVITY_PRINT_SETTINGS:
				if (resultCode == RESULT_OK) {
					Bundle extras = data.getExtras();
					if (extras != null) {
						_printSettings.put(TepraPrintParameterKey.Copies,
								extras.getInt(TepraPrintParameterKey.Copies));
						_printSettings.put(TepraPrintParameterKey.TapeCut,
								extras.getInt(TepraPrintParameterKey.TapeCut));
						_printSettings.put(TepraPrintParameterKey.HalfCut,
								extras.getBoolean(TepraPrintParameterKey.HalfCut));
						_printSettings.put(TepraPrintParameterKey.PrintSpeed,
								extras.getInt(TepraPrintParameterKey.PrintSpeed));
						_printSettings.put(TepraPrintParameterKey.Density,
								extras.getInt(TepraPrintParameterKey.Density));
						_printSettings.put(TepraPrintParameterKey.PriorityPrintSetting,
								extras.getBoolean(TepraPrintParameterKey.PriorityPrintSetting));
						_printSettings.put(TepraPrintParameterKey.HalfCutContinuous,
								extras.getBoolean(TepraPrintParameterKey.HalfCutContinuous));
					}
				}
				break;
			default:
				break;
		}
	}

	private Handler resultHandler = new Handler() {
		public void handleMessage(Message msg) {
			textResult.setText((String) msg.obj);
		}
	};

	public void printComplete(int connectionStatus, int status, boolean suspend) {
		String msg = "";
		if (connectionStatus == TepraPrintConnectionStatus.NoError && status == TepraPrintStatusError.NoError) {
			msg = "Print Complete.";
		} else {
			if (suspend) {
				msg = "Print Error Re-Print [" + Integer.toHexString(status)
						+ "].";
			} else {
				msg = "Print Error [" + Integer.toHexString(status) + "].";
			}
		}

		String title = getResources().getString(R.string.message_ticker);
		int iconId = R.drawable.nortification;
		Intent intent = new Intent(this, getClass());
		PendingIntent pendingIntent = PendingIntent.getActivity(this, 0, intent, PendingIntent.FLAG_IMMUTABLE);
		NotificationUtils notificationUtils = new NotificationUtils(this);
		Notification.Builder builder = notificationUtils.getNotification(pendingIntent, title, msg, iconId, true);
		notificationUtils.notify(NOTIFICATION_ID, builder);
	}

	private void alertAbortOperation(final String title, final String message) {
		handler.postDelayed(new Runnable() {
			public void run() {
				AlertDialog.Builder alert = new AlertDialog.Builder(
						MainActivity.this);
				alert.setTitle(title);
				alert.setMessage(message);
				alert.setPositiveButton("OK",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
							                    int which) {
								tepraPrint.cancelPrint();
							}
						});
				AlertDialog alertDialog = alert.create();
				alertDialog.setCanceledOnTouchOutside(false);
				alertDialog.show();
			}
		}, 1);
	}

	private void alertSuspendPrintOperation(final String title,
	                                        final String message) {
		handler.postDelayed(new Runnable() {
			public void run() {
				AlertDialog.Builder alert = new AlertDialog.Builder(
						MainActivity.this);
				alert.setTitle(title);
				alert.setMessage(message);
				alert.setPositiveButton("OK",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
							                    int which) {
								if (progressDialog != null) {
									progressDialog.show();
								}
								tepraPrint.resumeOfPrint();
							}
						});
				alert.setNegativeButton("Cancel",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
							                    int which) {
								tepraPrint.cancelPrint();
							}
						});
				AlertDialog alertDialog = alert.create();
				alertDialog.setCanceledOnTouchOutside(false);
				alertDialog.show();
			}
		}, 1);
	}

	private void adjustButtons(final boolean enabled) {
		handler.postDelayed(new Runnable() {
			public void run() {
				buttonSearch.setEnabled(enabled);
				buttonSelectXml.setEnabled(enabled);
				buttonPrintSettings.setEnabled(enabled);
				buttonPrint.setEnabled(enabled);
				buttonTapeCut.setEnabled(enabled);
				buttonTapeFeed.setEnabled(enabled);
			}
		}, 1);
	}

	private void runProgressDialogForPrinting() {
		handler.post(new Runnable() {
			public void run() {
				createProgressDialogForPrinting();
			}
		});
	}

	class SampleDataProvider implements TepraPrintDataProvider {

		private static final String DATA_DIR = "Sample";
		private static final String IMAGE_DIR = "Image";
		private static final String KEY_PREFFIX = "_CONTENTS";

		private String formName;

		List<Object> _formData = null;
		InputStream _formDataInputStream;

		List<ContentsData> _contentsData = null;

		public SampleDataProvider(String formName) {
			this.formName = formName;
			AssetManager as = getResources().getAssets();

			// Contents data
			String contentsFileName = TepraPrintSampleUtils.getPreffix(formName)
					+ KEY_PREFFIX + "." + TepraPrintSampleUtils.getSuffix(formName);
			TepraPrintContentsXmlParser xmlParser = new TepraPrintContentsXmlParser();
			InputStream in = null;
			try {
				in = as.open(DATA_DIR + "/" + contentsFileName);
				_contentsData = xmlParser.parse(in, "UTF-8");
			} catch (Exception e) {
				Logger.e(TAG, e.toString(), e);
			} finally {
				if (in != null) {
					try {
						in.close();
					} catch (IOException e) {
					}
					in = null;
				}
			}
		}

		public String getFormName() {
			return formName;
		}

		public void setFormName(String formName) {
			this.formName = formName;
		}

		@Override
		public void startOfPrint() {
			Logger.d(TAG, "startOfPrint");
		}

		@Override
		public void endOfPrint() {
			Logger.d(TAG, "endOfPrint");
		}

		@Override
		public void startPage() {
			Logger.d(TAG, "startPage");
		}

		@Override
		public void endPage() {
			Logger.d(TAG, "endPage");
		}

		@Override
		public int getNumberOfPages() {
			if (_contentsData == null) {
				Logger.d(TAG, "getNumberOfPages: 0");
				return 0;
			} else {
				Logger.d(TAG, "getNumberOfPages: "
						+ _contentsData.size());
				return _contentsData.size();
			}
		}

		@Override
		public InputStream getFormDataForPage(int pageIndex) {
			Logger.d(TAG, "getFormDataForPage: pageIndex="
					+ pageIndex);

			if (_formDataInputStream != null) {
				try {
					_formDataInputStream.close();
				} catch (IOException e) {
					Logger.e(TAG, e.toString(), e);
				}
				_formDataInputStream = null;
			}
			try {
				AssetManager as = getResources().getAssets();
				_formDataInputStream = as.open(DATA_DIR + "/" + formName);
				Logger.d(TAG, "getFormDataForPage: " + formName + "=" + _formDataInputStream.available());
			} catch (IOException e) {
				Logger.e(TAG, e.toString(), e);
			}

			return _formDataInputStream;
		}

		@Override
		public String getStringContentData(String contentName, int pageIndex) {
			Logger.d(TAG,
					"getStringContentData: contentName=" + contentName
							+ ", pageIndex=" + pageIndex);
			String content = null;
			if (_contentsData != null) {
				int index = pageIndex - 1;
				ContentsData pageDictionary = _contentsData.get(index);
				content = pageDictionary.get(contentName);
			}
			return content;
		}

		@Override
		public Bitmap getBitmapContentData(String contentName, int pageIndex) {
			Logger.d(TAG,
					"getBitmapContentData: contentName=" + contentName
							+ ", pageIndex=" + pageIndex);
			Bitmap bitmap = null;
			if (_contentsData != null) {
				int index = pageIndex - 1;
				ContentsData pageDictionary = _contentsData.get(index);
				String content = pageDictionary.get(contentName);
				InputStream is = null;
				try {
					is = getResources().getAssets().open(IMAGE_DIR + "/" + content);
					bitmap = BitmapFactory.decodeStream(is);
				} catch (IOException e) {
					Logger.e(TAG, e.toString(), e);
				} finally {
					if (is != null) {
						try {
							is.close();
						} catch (IOException e) {
						}
						is = null;
					}
				}
			}
			return bitmap;
		}

	}

	class PrintCallback implements TepraPrintCallback {

		@Override
		public void onChangePrintOperationPhase(TepraPrint tepraPrint, int phase) {
			Logger.d(TAG,
					"onChangePrintOperationPhase: phase=" + phase);
			waitDialog.dismiss();
			String jobPhase = "";
			switch (phase) {
				case TepraPrintPrintingPhase.Prepare:
					jobPhase = "PrintingPhasePrepare";
					break;
				case TepraPrintPrintingPhase.Processing:
					jobPhase = "PrintingPhaseProcessing";
					break;
				case TepraPrintPrintingPhase.WaitingForPrint:
					jobPhase = "PrintingPhaseWaitingForPrint";
					break;
				case TepraPrintPrintingPhase.Complete:
					jobPhase = "PrintingPhaseComplete";
					if (timer != null) {
						timer.cancel();
						timer = null;
					}
					if (progressDialog != null) {
						progressDialog.setProgress(0);
						progressDialog.dismiss();
						progressDialog = null;
					}
					runProgressDialogForPrinting();
					adjustButtons(true);

					_jobNumber++;
					performPrint();
					break;
				default:
					if (progressDialog != null) {
						progressDialog.setProgress(0);
						progressDialog.dismiss();
						progressDialog = null;
					}
					runProgressDialogForPrinting();
					adjustButtons(true);
					break;
			}
			Message msg = new Message();
			msg.obj = jobPhase;
			resultHandler.sendMessage(msg);
		}

		@Override
		public void onChangeTapeFeedOperationPhase(TepraPrint tepraPrint, int phase) {
			Logger.d(TAG,
					"onChangeTapeFeedOperationPhase: phase=" + phase);
			String jobPhase = "";
			switch (phase) {
				case TepraPrintPrintingPhase.Prepare:
					jobPhase = "PrintingPhasePrepare";
					break;
				case TepraPrintPrintingPhase.Processing:
					jobPhase = "PrintingPhaseProcessing";
					break;
				case TepraPrintPrintingPhase.WaitingForPrint:
					jobPhase = "PrintingPhaseWaitingForPrint";
					break;
				case TepraPrintPrintingPhase.Complete:
					jobPhase = "PrintingPhaseComplete";
					waitDialog.dismiss();
					adjustButtons(true);
					_processing = false;
					break;
				default:
					waitDialog.dismiss();
					adjustButtons(true);
					break;
			}
			Message msg = new Message();
			msg.obj = jobPhase;
			resultHandler.sendMessage(msg);
		}

		@Override
		public void onAbortPrintOperation(TepraPrint tepraPrint, int errorStatus,
		                                  int deviceStatus) {
			Logger.d(TAG,
					"onAbortPrintOperation: errorStatus=" + errorStatus
							+ ", deviceStatus=" + deviceStatus);
			waitDialog.dismiss();
			printComplete(errorStatus, deviceStatus, false);

			if (timer != null) {
				timer.cancel();
				timer = null;
			}
			if (progressDialog != null) {
				progressDialog.setProgress(0);
				progressDialog.dismiss();
				progressDialog = null;
			}
			runProgressDialogForPrinting();
			_processing = false;

			String message = "Error Status : " + errorStatus
					+ "\nDevice Status : " + Integer.toHexString(deviceStatus);
			Message msg = new Message();
			msg.obj = message;
			resultHandler.sendMessage(msg);

			alertAbortOperation("Print Error!", message);
		}

		@Override
		public void onSuspendPrintOperation(TepraPrint tepraPrint, int errorStatus,
		                                    int deviceStatus) {
			Logger.d(TAG,
					"onSuspendPrintOperation: errorStatus=" + errorStatus
							+ ", deviceStatus=" + deviceStatus);
			waitDialog.dismiss();
			printComplete(errorStatus, deviceStatus, true);
			if (progressDialog != null) {
				progressDialog.setProgress(0);
				progressDialog.dismiss();
				progressDialog = null;
			}
			runProgressDialogForPrinting();

			String message = "Error Status : " + errorStatus
					+ "\nDevice Status : " + Integer.toHexString(deviceStatus);
			Message msg = new Message();
			msg.obj = message;
			resultHandler.sendMessage(msg);

			alertSuspendPrintOperation("Print Error! re-print ?", message);
		}

		@Override
		public void onAbortTapeFeedOperation(TepraPrint tepraPrint, int errorStatus,
		                                     int deviceStatus) {
			Logger.d(TAG,
					"onAbortTapeFeedOperation: errorStatus=" + errorStatus
							+ ", deviceStatus=" + deviceStatus);
			adjustButtons(true);

			if (timer != null) {
				timer.cancel();
				timer = null;
			}
			waitDialog.dismiss();
			_processing = false;

			String message = "Error Status : " + errorStatus
					+ "\nDevice Status : " + Integer.toHexString(deviceStatus);
			Message msg = new Message();
			msg.obj = message;
			resultHandler.sendMessage(msg);

			alertAbortOperation("Tape Feed Error!", message);
		}

	}

	private boolean isSupportPriorityPrint() {
		boolean supportPriorityPrint = false;

		if (_printerInfo != null) {
			// getModelName
			tepraPrint.setPrinterInformation(_printerInfo);
			String modelName = tepraPrint.getModelName();

			if ("TEPRA PRO SR-R7900P".equals(modelName) || "SR-MK1".equals(modelName) || "SR-R2500P".equals(modelName)) {
				supportPriorityPrint = true;
			}
		}

		return supportPriorityPrint;
	}

	private boolean isSupportHalfCutContinuous() {
		boolean supportHalfCutContinuous = false;

		if (_printerInfo != null) {
			// getModelName
			tepraPrint.setPrinterInformation(_printerInfo);
			String modelName = tepraPrint.getModelName();

			if ("TEPRA PRO SR-R7900P".equals(modelName) || "TEPRA PRO SR5900P".equals(modelName) || "TEPRA PRO SR5900C".equals(modelName) || "TEPRA PRO SR5900GS".equals(modelName)) {
				supportHalfCutContinuous = true;
			}
		}

		return supportHalfCutContinuous;
	}

}
