/*
 * MainActivity.java
 *
 * Project: TEPRA-Print SDK
 *
 * Contains: MainActivity class
 *           SampleDataProvider class
 *           PrintCallback class
 *
 * (C) 2016-2019 KING JIM CO.,LTD.
 */
package jp.co.kingjim.tepraprint.sdk.simpledemo;

import static java.lang.Math.ceil;

import android.Manifest;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Notification;
import android.app.PendingIntent;
import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.KeyEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnKeyListener;
import android.view.WindowManager;
import android.view.inputmethod.InputMethodManager;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;
import android.widget.Toast;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import jp.co.kingjim.tepraprint.sdk.TepraPrint;
import jp.co.kingjim.tepraprint.sdk.TepraPrintCallback;
import jp.co.kingjim.tepraprint.sdk.TepraPrintConnectionStatus;
import jp.co.kingjim.tepraprint.sdk.TepraPrintDataProvider;
import jp.co.kingjim.tepraprint.sdk.TepraPrintDiscoverPrinter;
import jp.co.kingjim.tepraprint.sdk.TepraPrintParameterKey;
import jp.co.kingjim.tepraprint.sdk.TepraPrintPrintSpeed;
import jp.co.kingjim.tepraprint.sdk.TepraPrintPrintingPhase;
import jp.co.kingjim.tepraprint.sdk.TepraPrintStatusError;
import jp.co.kingjim.tepraprint.sdk.TepraPrintTapeCut;
import jp.co.kingjim.tepraprint.sdk.TepraPrintTapeWidth;

public class MainActivity extends Activity implements OnClickListener {

//	private Object ArrayAdapter;

	public enum FormType {
		String,
		QRCode
	}

	private final String TAG = getClass().getSimpleName();

	private static final int NOTIFICATION_ID = 1;

	private static final int REQUEST_ENABLE_BT = 0;
	private static final int REQUEST_ACTIVITY_SEARCH = 1;

	private static final int REQUEST_PERMISSIONS = 1;

	TepraPrint tepraPrint;
	PrintCallback printListener;

	SampleDataProvider sampleDataProvider;

	private Button buttonDiscoverPrinter;
	private EditText editTextInputData;
	private Button buttonPreview;
	private Button buttonPrint;
	private Spinner spinnerData;
	private Spinner spinnerMargin;

	private boolean processing = false;

	Map<String, String> printerInfo = null;
	Map<String, Integer> tepraStatus = null;

	android.os.Handler handler = new android.os.Handler();
	static Bitmap mBitmap = null;
	static ArrayList<Bitmap> mBitmaps = null;
	Context mContext = null;
	int mTapeWidth = 0;
	int prePosition = 0;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);

		mContext = this;

		// Keep screen on
		getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);

		tepraPrint = new TepraPrint(this);
		// Sets the callback
		tepraPrint.setCallback(printListener = new PrintCallback());

		sampleDataProvider = new SampleDataProvider();

		buttonDiscoverPrinter = (Button) findViewById(R.id.button_discover_printer);
		buttonDiscoverPrinter.setOnClickListener(this);

		editTextInputData = (EditText) findViewById(R.id.edittext_input_data);
		editTextInputData.clearFocus();
		editTextInputData.setOnKeyListener(new OnKeyListener() {
			public boolean onKey(View v, int keyCode, KeyEvent event) {
				if (event.getAction() == KeyEvent.ACTION_DOWN
						&& keyCode == KeyEvent.KEYCODE_ENTER) {
					onEnter();
					return true;
				}
				return false;
			}
		});
		editTextInputData.setText(sampleDataProvider.getStringData());

		buttonPrint = (Button) findViewById(R.id.button_print);
		buttonPrint.setOnClickListener(this);

		buttonPreview = (Button) findViewById(R.id.button_preview);
		buttonPreview.setOnClickListener(this);

		// Bluetooth
		enableBluetooth();

		spinnerData = (Spinner) findViewById(R.id.spinner_data);
		String[] data = getResources().getStringArray(R.array.Data_List);

		ArrayAdapter<String> adapter = new ArrayAdapter<>(
				this,
				android.R.layout.simple_spinner_item,
				data
		);
		adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);

		spinnerMargin = (Spinner) findViewById(R.id.spinner_margin);

		spinnerData.setAdapter(adapter);
		spinnerData.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
			@Override
			public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
				Spinner spinnerData = (Spinner) parent;
				String item = (String) spinnerData.getSelectedItem();
				String inputData = editTextInputData.getText().toString();
				String data = "";

				if (mBitmap != null) {
					mBitmap.recycle();
					mBitmap = null;
				}

				if(mBitmaps != null){
					for (int i = 0; i < mBitmaps.size(); i++) {
						mBitmaps.get(i).recycle();
					}
					mBitmaps.clear();
					mBitmaps = null;
				}

				spinnerMargin.setVisibility(View.VISIBLE);
				switch (item) {
					case "Text":
						if(prePosition == 1) {
							sampleDataProvider.setQrCodeData(inputData);
						}
						data = sampleDataProvider.getStringData();
						spinnerMargin.setVisibility(View.INVISIBLE);
						break;
					case "QRCode":
						if(prePosition == 0) {
							sampleDataProvider.setStringData(inputData);
						}
						data = sampleDataProvider.getQrCodeData();
						spinnerMargin.setVisibility(View.INVISIBLE);
						break;
					case "Img1":
						break;
					case "Img2":
						break;
					case "Imgs":
						editTextInputData.setText("");
						break;
					default:
						break;
				}
				editTextInputData.setText(data);
				prePosition = position;
			}

			@Override
			public void onNothingSelected(AdapterView<?> adapterView) {

			}

		});

	}

	private void enableBluetooth() {
		try {
			BluetoothAdapter btAdapter = BluetoothAdapter.getDefaultAdapter();
			if (btAdapter == null) {
				Toast.makeText(getApplicationContext(),
						"Bluetooth is not available.", Toast.LENGTH_SHORT).show();
			} else {
				if (!btAdapter.isEnabled()) {
					Intent intent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
					if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
						if (checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT) == PackageManager.PERMISSION_GRANTED) {
							startActivityForResult(intent, REQUEST_ENABLE_BT);
						}
					} else {
						startActivityForResult(intent, REQUEST_ENABLE_BT);
					}
				}
			}
		} catch (Exception e) {
			Logger.w(TAG, "", e);
		}
	}

	public static Bitmap loadAssetsBitmap(Context context, String name) {
		Bitmap bmpResult = null;
		InputStream is = null;

		try {
			is = context.getAssets().open(name);
			bmpResult = BitmapFactory.decodeStream(is);
		} catch (IOException e) {
			Logger.e(e.toString(), e);
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {
					Logger.e(e.toString(), e);
				}
			}
		}

		return bmpResult;
	}

	@Override
	protected void onStart() {
		super.onStart();

		ArrayList<String> permissions = new ArrayList<>();

		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) { // API33 or higher
			// BLUETOOTH AUTH = BLUETOOTH_CONNECT
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_CONNECT);
			}
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_SCAN) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_SCAN);
			}
			// WIFI AUTH = NEARBY_WIFI_DEVICES
			if (checkSelfPermission(Manifest.permission.NEARBY_WIFI_DEVICES) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.NEARBY_WIFI_DEVICES);
			}
			// NOTIFICATION AUTH
			if (checkSelfPermission(Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.POST_NOTIFICATIONS);
			}
		} else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) { // API31,32
			// BLUETOOTH AUTH = BLUETOOTH_CONNECT
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_CONNECT) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_CONNECT);
			}
			if (checkSelfPermission(Manifest.permission.BLUETOOTH_SCAN) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.BLUETOOTH_SCAN);
			}
			// WIFI AUTH = LOCATION
			if (checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_FINE_LOCATION);
			}
			if (checkSelfPermission(Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_COARSE_LOCATION);
			}
		} else { // API30 or lower
			// BLUETOOTH AUTH & WIFI AUTH = LOCATION
			if (checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_FINE_LOCATION);
			}
			if (checkSelfPermission(Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
				permissions.add(Manifest.permission.ACCESS_COARSE_LOCATION);
			}
		}

		if (!permissions.isEmpty()) {
			requestPermissions(permissions.toArray(new String[0]), REQUEST_PERMISSIONS);
		}
	}

	@Override
	public void onRequestPermissionsResult(int requestCode,
										   String[] permissions, int[] grantResults) {
		switch (requestCode) {
			case REQUEST_PERMISSIONS: {
				boolean isGranted = true;
				for (int grantResult : grantResults) {
					if (grantResult != PackageManager.PERMISSION_GRANTED) {
						isGranted = false;
						break;
					}
				}
				if (!isGranted) {
					Toast.makeText(getApplicationContext(),
							"Since permission has not been granted, this app will not be able to discover the Bluetooth printer.", Toast.LENGTH_SHORT).show();
				}
				break;
			}
		}
	}

	private void onEnter(){
		setInputDataToDataProvider();

		editTextInputData.clearFocus();

		hideKeyboard();
	}

	private void setInputDataToDataProvider() {
		String inputData = editTextInputData.getText().toString();
		String item = spinnerData.getSelectedItem().toString();
		switch (item) {
		case "Text":
			sampleDataProvider.setStringData(inputData);
			break;
		case "QRCode":
			sampleDataProvider.setQrCodeData(inputData);
			break;
		default:
			break;
		}
	}

	private void hideKeyboard() {
		// Check if no view has focus:
		View view = this.getCurrentFocus();
		if (view != null) {
			InputMethodManager inputMethodManager = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
			inputMethodManager.hideSoftInputFromWindow(view.getWindowToken(), InputMethodManager.HIDE_NOT_ALWAYS);
		}
	}

	@Override
	public void onClick(View v) {
		Intent intent = null;

		setInputDataToDataProvider();

		editTextInputData.clearFocus();

		hideKeyboard();

		int viewId = v.getId();
		if (viewId == R.id.button_discover_printer) {
			intent = new Intent(this, SearchActivity.class);
			startActivityForResult(intent, REQUEST_ACTIVITY_SEARCH);
		} else if (viewId == R.id.button_print) {
			if (printerInfo == null) {
				Toast.makeText(getApplicationContext(),
						"Device is not selected.", Toast.LENGTH_SHORT).show();
			} else {
				setProcessing(true);
				performPrint();
			}
		} else if (viewId == R.id.button_preview) {
			if (printerInfo == null) {
				Toast.makeText(getApplicationContext(), "Device is not selected.", Toast.LENGTH_SHORT).show();
			} else {
				setProcessing(true);
				performPreview();
			}
		}
	}

	public boolean isProcessing() {
		return processing;
	}

	private void setProcessing(final Boolean mode) {
		processing = mode;

		handler.postDelayed(new Runnable() {
			public void run() {
				buttonDiscoverPrinter.setEnabled(!mode);
				editTextInputData.setEnabled(!mode);
				buttonPrint.setEnabled(!mode);
				buttonPreview.setEnabled(!mode);
				spinnerData.setEnabled(!mode);
				spinnerMargin.setEnabled(!mode);

			}
		}, 1);
	}

	private void performPrint() {
		if (printerInfo == null) {
			setProcessing(false);
			return;
		}

		final Context self = this;

		ExecutorService executor = Executors.newSingleThreadExecutor();
		executor.execute(new Runnable() {
			@Override
			public void run() {
				Boolean printResult;

				// Set printing information
				tepraPrint.setPrinterInformation(printerInfo);

				// Obtain printing status
				tepraStatus = tepraPrint.fetchPrinterStatus();
				int deviceError = tepraPrint.getDeviceErrorFromStatus(tepraStatus);
				if (tepraStatus.isEmpty() || (deviceError == TepraPrintStatusError.ConnectionFailed)) {
					printResult = false;
				} else {
					// Make a print parameter
					int tapeWidth = tepraPrint.getTapeWidthFromStatus(tepraStatus);

					Map<String, Object> printParameter = new HashMap<String, Object>();
					// Number of copies(1 ... 99)
					printParameter.put(TepraPrintParameterKey.Copies, 1);
					// Tape cut method(TepraPrintTapeCut)
					printParameter.put(TepraPrintParameterKey.TapeCut, TepraPrintTapeCut.EachLabel);
					// Set half cut (true:half cut on)
					printParameter.put(TepraPrintParameterKey.HalfCut, tepraPrint.isSupportHalfCut());
					// Low speed print setting (true:low speed print on)
					printParameter.put(TepraPrintParameterKey.PrintSpeed, TepraPrintPrintSpeed.PrintSpeedHigh);
					// Print density(-5 ... 5)
					printParameter.put(TepraPrintParameterKey.Density, 0);
					// Tape width(TepraPrintTapeWidth)
					printParameter.put(TepraPrintParameterKey.TapeWidth, tapeWidth);
					// Prioritize print settings
					printParameter.put(TepraPrintParameterKey.PriorityPrintSetting, false);
					// Half cut continuous setting
					printParameter.put(TepraPrintParameterKey.HalfCutContinuous, false);

					String item = spinnerData.getSelectedItem().toString();
					int margin = Integer.parseInt(spinnerMargin.getSelectedItem().toString());

					switch (item) {
						case "Text":
							sampleDataProvider.setFormType(FormType.String);
							tepraPrint.doPrint(sampleDataProvider, printParameter);
							break;
						case "QRCode":
							sampleDataProvider.setFormType(FormType.QRCode);
							tepraPrint.doPrint(sampleDataProvider, printParameter);
							break;
						case "Img1":
							Bitmap bitmap1 = createBitmap("Hello Tape:1", margin);
							tepraPrint.doPrint(bitmap1, printParameter);
							break;
						case "Img2":
							Bitmap bitmap2 = createBitmap("Hello Tape:2", margin);
							tepraPrint.doPrint(bitmap2, printParameter);
							break;
						case "Imgs":
							Bitmap bitmap3 = createBitmap("Hello Tape:1", margin);
							Bitmap bitmap4 = createBitmap("Hello Tape:2", margin);

							if (bitmap3 != null && bitmap4 != null) {
								ArrayList<Bitmap> bitmaps = new ArrayList<Bitmap>(
										Arrays.asList(bitmap3, bitmap4)
								);
								tepraPrint.doPrint(bitmaps, printParameter);
							}
							break;
						default:
							break;
					}

					printResult = true;
				}

				final Boolean result = printResult;
				new Handler(Looper.getMainLooper()).post(new Runnable() {
					@Override
					public void run() {
						if (result == false) {
							setProcessing(false);

							String message = "Can't get printer status.";
							alertAbortOperation("Error", message);
						}
					}
				});
			}
		});
	}

	private void performPreview() {
		if (printerInfo == null) {
			setProcessing(false);
			return;
		}

		ExecutorService executor = Executors.newSingleThreadExecutor();
		executor.execute(new Runnable() {
			@Override
			public void run() {
				// Set printing information
				tepraPrint.setPrinterInformation(printerInfo);

				// Obtain printing status
				mTapeWidth = TepraPrintTapeWidth.Normal_18mm;
				tepraStatus = tepraPrint.fetchPrinterStatus();
				int deviceError = tepraPrint.getDeviceErrorFromStatus(tepraStatus);
				if (!tepraStatus.isEmpty() && (deviceError != TepraPrintStatusError.ConnectionFailed)) {
					mTapeWidth = tepraPrint.getTapeWidthFromStatus(tepraStatus);
				}

				final Boolean result = true;
				new Handler(Looper.getMainLooper()).post(new Runnable() {
					@Override
					public void run() {
						setProcessing(false);

						if (result == true) {
							// Make a print parameter
							Map<String, Object> printParameter = new HashMap<String, Object>();
							// Number of copies(1 ... 99)
							printParameter.put(TepraPrintParameterKey.Copies, 1);
							// Tape cut method(TepraPrintTapeCut)
							printParameter.put(TepraPrintParameterKey.TapeCut, TepraPrintTapeCut.EachLabel);
							// Set half cut (true:half cut on)
							printParameter.put(TepraPrintParameterKey.HalfCut, true);
							// Low speed print setting (true:low speed print on)
							printParameter.put(TepraPrintParameterKey.PrintSpeed, TepraPrintPrintSpeed.PrintSpeedHigh);
							// Print density(-5 ... 5)
							printParameter.put(TepraPrintParameterKey.Density, 0);
							// Tape width(TepraPrintTapeWidth)
							printParameter.put(TepraPrintParameterKey.TapeWidth, mTapeWidth);
							// Prioritize print settings
							printParameter.put(TepraPrintParameterKey.PriorityPrintSetting, false);
							// Half cut continuous setting
							printParameter.put(TepraPrintParameterKey.HalfCutContinuous, false);

							String item = spinnerData.getSelectedItem().toString();
							int margin = Integer.parseInt(spinnerMargin.getSelectedItem().toString());
							List<Bitmap> images = null;
							switch (item) {
								case "Text":
									sampleDataProvider.setFormType(FormType.String);
									images = tepraPrint.getLabelImages(sampleDataProvider, printParameter);
									if (images.size() > 0) {
										// first page
										mBitmap = images.get(0);
									}
									break;
								case "QRCode":
									sampleDataProvider.setFormType(FormType.QRCode);
									images = tepraPrint.getLabelImages(sampleDataProvider, printParameter);
									if (images.size() > 0) {
										// first page
										mBitmap = images.get(0);
									}
									break;
								case "Img1":
									mBitmap = createBitmap("Hello Tape:1", margin);
									break;
								case "Img2":
									mBitmap = createBitmap("Hello Tape:2", margin);
									break;
								case "Imgs":
									mBitmap = createBitmap("Hello Tape:1", margin);
									break;
								default:
									break;
							}

							Intent intent = new Intent(mContext, PreviewActivity.class);
							startActivity(intent);
						} else {
							String message = "Can't get printer status.";
							alertAbortOperation("Error", message);
						}
					}

				});
			}
		});
	}

	@Override
	public void onDestroy() {
		sampleDataProvider.closeStreams();

		// Keep screen off
		getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
		super.onDestroy();
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);
		switch (requestCode) {
		case REQUEST_ENABLE_BT:
			if (resultCode != RESULT_OK) {
				Toast.makeText(getApplicationContext(),
						"Bluetooth is not enabled.", Toast.LENGTH_SHORT).show();
			}
			break;
		case REQUEST_ACTIVITY_SEARCH:
			if (resultCode == RESULT_OK) {
				Bundle extras = data.getExtras();
				if (extras != null) {
					if (printerInfo != null) {
						printerInfo.clear();
						printerInfo = null;
					}
					printerInfo = new HashMap<String, String>();
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_NAME,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_NAME));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_PRODUCT,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_PRODUCT));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_USBMDL,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_USBMDL));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_HOST,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_HOST));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_PORT,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_PORT));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_TYPE,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_TYPE));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_DOMAIN,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_DOMAIN));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_SERIAL_NUMBER,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_SERIAL_NUMBER));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_CLASS,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_CLASS));
					printerInfo
							.put(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_STATUS,
									extras.getString(TepraPrintDiscoverPrinter.PRINTER_INFO_DEVICE_STATUS));
					String s = getResources().getString( R.string.button_search) + extras.getString("name");
					buttonDiscoverPrinter.setText(s);
				}
			}
			break;
		default:
			break;
		}
	}

	private Bitmap createBitmap(String text, int marginValue) {
		if (printerInfo == null || tepraStatus == null) {
			return null;
		}

		tepraPrint.setPrinterInformation(printerInfo);
		int resolution = tepraPrint.getResolution();
		float margin = (float)marginValue / 25.4f * (float)resolution;
		int tapeWidth = tepraPrint.getTapeWidthFromStatus(tepraStatus);
		// tapeWidth が正常に取得できない時は デフォルトのテープ幅として 12mm を設定する
		if (tapeWidth == TepraPrintTapeWidth.None || tapeWidth == TepraPrintTapeWidth.Unknown) {
			tapeWidth = TepraPrintTapeWidth.Normal_12mm;
		}
		int height = tepraPrint.getPrintableSizeFromTape(tapeWidth);
		float ovalSize = (float)height / 2f;

		Paint paint = new Paint();
		paint.setTextSize(height * 0.8f);
		float textSize = paint.measureText(text);
		int width = (int)(ceil(margin) + ovalSize * 2f + textSize + ceil(margin));

		Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
		Canvas canvas = new Canvas(bitmap);
		paint.setColor(Color.WHITE);
		paint.setStyle(Paint.Style.FILL);
		canvas.drawRect(0f, 0f, width, height, paint);
		paint.setColor(Color.BLACK);
		paint.setStyle(Paint.Style.STROKE);
		canvas.drawCircle(margin + ovalSize, ovalSize, ovalSize, paint);
		paint.setStyle(Paint.Style.FILL);
		canvas.drawText(text, margin + ovalSize * 2, height - (height * 0.25f), paint);

		return bitmap;
	}



	public void printComplete(int connectionStatus, int status, boolean suspend) {
		String msg = "";
		if (connectionStatus == TepraPrintConnectionStatus.NoError && status == TepraPrintStatusError.NoError) {
			msg = "Print Complete.";
		} else {
			if (suspend) {
				msg = "Print Error Re-Print [" + Integer.toHexString(status)
						+ "].";
			} else {
				msg = "Print Error [" + Integer.toHexString(status) + "].";
			}
		}

		String title = getResources().getString(R.string.message_ticker);
		int iconId = R.drawable.nortification;
		Intent intent = new Intent(this, getClass());
		PendingIntent pendingIntent = PendingIntent.getActivity(this, 0, intent, PendingIntent.FLAG_IMMUTABLE);
		NotificationUtils notificationUtils = new NotificationUtils(this);
		Notification.Builder builder = notificationUtils.getNotification(pendingIntent, title, msg, iconId, true);
		notificationUtils.notify(NOTIFICATION_ID, builder);
	}

	private void alertAbortOperation(final String title, final String message) {
		handler.postDelayed(new Runnable() {
			public void run() {
				AlertDialog.Builder alert = new AlertDialog.Builder(
						MainActivity.this);
				alert.setTitle(title);
				alert.setMessage(message);
				alert.setPositiveButton("OK",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
									int which) {
								tepraPrint.cancelPrint();
							}
						});
				AlertDialog alertDialog = alert.create();
				alertDialog.setCanceledOnTouchOutside(false);
				alertDialog.show();
			}
		}, 1);
	}

	private void alertSuspendPrintOperation(final String title,
			final String message) {
		handler.postDelayed(new Runnable() {
			public void run() {
				AlertDialog.Builder alert = new AlertDialog.Builder(
						MainActivity.this);
				alert.setTitle(title);
				alert.setMessage(message);
				alert.setPositiveButton("OK",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
									int which) {
								tepraPrint.resumeOfPrint();
							}
						});
				alert.setNegativeButton("Cancel",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
									int which) {
								tepraPrint.cancelPrint();
							}
						});
				AlertDialog alertDialog = alert.create();
				alertDialog.setCanceledOnTouchOutside(false);
				alertDialog.show();
			}
		}, 1);
	}

	class SampleDataProvider implements TepraPrintDataProvider {

		private static final String FORM_DATA_STRING = "FormDataString.plist";
		private static final String FORM_DATA_QRCODE = "FormDataQRCode.plist";

		private FormType formType = FormType.String;
		private String stringData = "String";
		private String qrCodeData = "QRCode";

		InputStream formDataStringInputStream;
		InputStream formDataQRCodeInputStream;

		public FormType getFormType() {
			return formType;
		}

		public void setFormType(FormType formType) {
			this.formType = formType;
		}

		public String getStringData() {
			return stringData;
		}

		public void setStringData(String stringData) {
			this.stringData = stringData;
		}

		public String getQrCodeData() {
			return qrCodeData;
		}

		public void setQrCodeData(String qrCodeData) {
			this.qrCodeData = qrCodeData;
		}

		public void closeStreams() {
			if (formDataStringInputStream != null) {
				try {
					formDataStringInputStream.close();
				} catch (IOException e) {
					Logger.e(e.toString(), e);
				}
				formDataStringInputStream = null;
			}
			if (formDataQRCodeInputStream != null) {
				try {
					formDataQRCodeInputStream.close();
				} catch (IOException e) {
					Logger.e(e.toString(), e);
				}
				formDataQRCodeInputStream = null;
			}
		}

		@Override
		public void startOfPrint() {
			// It is called only once when printing started
			Logger.d("startOfPrint");
		}

		@Override
		public void endOfPrint() {
			// It is called only once when printing finished
			Logger.d("endOfPrint");
		}

		@Override
		public void startPage() {
			// It is called when starting a page
			Logger.d("startPage");
		}

		@Override
		public void endPage() {
			// It is called when finishing a page
			Logger.d("endPage");
		}

		@Override
		public int getNumberOfPages() {
			// Return all pages printed
			Logger.d("getNumberOfPages");

			return 1;
		}

		@Override
		public InputStream getFormDataForPage(int pageIndex) {
			// Return the form data for pageIndex page
			Logger.d("getFormDataForPage: pageIndex=" + pageIndex);

			InputStream formData = null;

			switch (formType) {
			case String:
				Logger.d("Stinrg: pageIndex=" + pageIndex);
				if (formDataStringInputStream != null) {
					try {
						formDataStringInputStream.close();
					} catch (IOException e) {
						Logger.e(e.toString(), e);
					}
					formDataStringInputStream = null;
				}
				try {
					AssetManager as = getResources().getAssets();
					formDataStringInputStream = as.open(FORM_DATA_STRING);
					formData = formDataStringInputStream;
					Logger.d("getFormDataForPage: " + FORM_DATA_STRING + "=" + formDataStringInputStream.available());
				} catch (IOException e) {
					Logger.e(e.toString(), e);
				}
				break;
			case QRCode:
				Logger.d("QRCode: pageIndex=" + pageIndex);
				if (formDataQRCodeInputStream != null) {
					try {
						formDataQRCodeInputStream.close();
					} catch (IOException e) {
						Logger.e(e.toString(), e);
					}
					formDataQRCodeInputStream = null;
				}
				try {
					AssetManager as = getResources().getAssets();
					formDataQRCodeInputStream = as.open(FORM_DATA_QRCODE);
					formData = formDataQRCodeInputStream;
					Logger.d("getFormDataForPage: " + FORM_DATA_QRCODE + "=" + formDataStringInputStream.available());
				} catch (IOException e) {
					Logger.e(e.toString(), e);
				}
				break;
			}

			return formData;
		}

		@Override
		public String getStringContentData(String contentName, int pageIndex) {
			// Return the data for the contentName of the pageIndex page
			Logger.d("getStringContentData: contentName=" + contentName
							+ ", pageIndex=" + pageIndex);

			if ("String".equals(contentName)) {
				return stringData;
			} else if ("QRCode".equals(contentName)) {
				return qrCodeData;
			}

			return null;
		}

		@Override
		public Bitmap getBitmapContentData(String contentName, int pageIndex) {
			// Return the data for the contentName of the pageIndex page
			Logger.d("getBitmapContentData: contentName=" + contentName
							+ ", pageIndex=" + pageIndex);

			return null;
		}

	}

	class PrintCallback implements TepraPrintCallback {

		@Override
		public void onChangePrintOperationPhase(TepraPrint tepraPrint, int phase) {
			// Report the change of a printing phase
			Logger.d("onChangePrintOperationPhase: phase=" + phase);
			String jobPhase = "";
			switch (phase) {
			case TepraPrintPrintingPhase.Prepare:
				jobPhase = "PrintingPhasePrepare";
				break;
			case TepraPrintPrintingPhase.Processing:
				jobPhase = "PrintingPhaseProcessing";
				break;
			case TepraPrintPrintingPhase.WaitingForPrint:
				jobPhase = "PrintingPhaseWaitingForPrint";
				break;
			case TepraPrintPrintingPhase.Complete:
				jobPhase = "PrintingPhaseComplete";
				printComplete(TepraPrintConnectionStatus.NoError, TepraPrintStatusError.NoError, false);
				setProcessing(false);
				break;
			default:
				setProcessing(false);
				break;
			}
			Logger.d("phase=" + jobPhase);
		}

		@Override
		public void onChangeTapeFeedOperationPhase(TepraPrint tepraPrint, int phase) {
			// Called when tape feed and tape cutting state transitions
			Logger.d("onChangeTapeFeedOperationPhase: phase=" + phase);
		}

		@Override
		public void onAbortPrintOperation(TepraPrint tepraPrint, int errorStatus,
				int deviceStatus) {
			// It is called when undergoing a transition to the printing cancel operation due to a printing error
			Logger.d("onAbortPrintOperation: errorStatus=" + errorStatus
							+ ", deviceStatus=" + deviceStatus);

			printComplete(errorStatus, deviceStatus, false);

			setProcessing(false);

			String message = "Error Status : " + errorStatus
					+ "\nDevice Status : " + Integer.toHexString(deviceStatus);
			alertAbortOperation("Print Error!", message);
		}

		@Override
		public void onSuspendPrintOperation(TepraPrint tepraPrint, int errorStatus,
				int deviceStatus) {
			// It is called when undergoing a transition to the printing restart operation due to a printing error
			Logger.d("onSuspendPrintOperation: errorStatus=" + errorStatus
							+ ", deviceStatus=" + deviceStatus);

			printComplete(errorStatus, deviceStatus, true);

			String message = "Error Status : " + errorStatus
					+ "\nDevice Status : " + Integer.toHexString(deviceStatus);
			alertSuspendPrintOperation("Print Error! re-print ?", message);
		}

		@Override
		public void onAbortTapeFeedOperation(TepraPrint tepraPrint, int errorStatus,
				int deviceStatus) {
			// Called when tape feed and tape cutting stops due to an error
			Logger.d("errorStatus=" + errorStatus + ", deviceStatus=" + deviceStatus);
		}

	}

}
